package ru.yandex.tours.prices

import org.joda.time.format.DateTimeFormat
import org.joda.time.{DateTime, LocalDate}
import ru.yandex.tours.calendar.Calendar.FlightDay
import ru.yandex.tours.clickhouse.ClickHouseClient
import ru.yandex.tours.model.filter.snippet._
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.filter.SnippetFilter
import ru.yandex.tours.model.search.SearchDates
import ru.yandex.tours.model.util.{DateInterval, Nights}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.parsing.{IntValue, LocalDateValue}
import ru.yandex.tours.util.lang.Dates._

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 05.08.15
 */
class ClickHousePriceSearcher(clickHouseClient: ClickHouseClient, table: String = "price_history")
                             (implicit ec: ExecutionContext) extends PriceSearcher with Logging {

  require(table.matches("[A-Za-z0-9_]+"), "table name should match pattern")

  private val dateFormat = DateTimeFormat.forPattern("yyyy-MM-dd")
  private val dateTimeFormat = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss")

  private def dateIn(column: String, dateInterval: DateInterval) = {
    import dateInterval._
    s"($column >= \'${start.toString(dateFormat)}\') AND ($column <= \'${end.toString(dateFormat)}\')"
  }

  private def nightsIn(column: String, nights: Nights) = {
    if (nights.values.isEmpty) "(1 = 1)"
    else s"($column in ${nights.values.mkString("(", ",", ")")})"
  }

  private def intValueIn(column: String, values: Iterable[Int]) = {
    if (values.isEmpty) "(1 = 1)"
    else s"($column in ${values.mkString("(", ",", ")")})"
  }

  private def margin(str: String, spaces: Int) = {
    str.split("\n").map(" " * spaces + _).mkString("\n")
  }

  case class RawPrice(raw: Price, isDirectionSearch: Boolean)

  private def cleanPrices(rawPrices: Seq[RawPrice]): Seq[Price] = {
    rawPrices.groupBy(p => (p.raw.when, p.raw.nights)).flatMap {
      case (_, prices) =>
        val (directionPrices, hotelPrices) = prices.partition(_.isDirectionSearch)
        if (directionPrices.nonEmpty) {
          val lastDirectionSearch = directionPrices.maxBy(_.raw.created).raw
          (hotelPrices ++ directionPrices).map(_.raw).filter(_.created >= lastDirectionSearch.created)
        } else {
          hotelPrices.map(_.raw)
        }
    }.filter(_.price > 0).toVector
  }

  def searchLastPrices(from: Int, to: Int, ages: Seq[Int],
                       dates: SearchDates,
                       filters: Seq[SnippetFilter],
                       freshness: FiniteDuration): Future[Seq[Price]] = {

    val whereQuery = filters.collect {
      case HotelIdFilter(Some(hotelId)) => s"(hotel_id = $hotelId)"
      case PansionFilter(pansions) => intValueIn("ps.pansion", pansions.map(_.getNumber))
      case SearchSourceFilter(TourOperatorFilter.name, operators) => intValueIn("ps.operator_id", operators)
      case PriceFilter(fromPrice, toPrice) if fromPrice.isDefined || toPrice.isDefined =>
        Seq(
          fromPrice.map(p => "ps.price >= " + p),
          toPrice.map(p => "ps.price <= " + p)
        ).flatten.mkString(" AND ")
    }.map("  AND " + _).mkString("\n")

    val query =
      s"""SELECT
          |    hotel_id,
          |    when,
          |    nights,
          |    when + argMax(nights, timestamp) AS when_back,
          |    argMax(`ps.price`, timestamp) AS price_last,
          |    argMax(`ps.operator_id`, timestamp) AS operator_last,
          |    argMax(`ps.pansion`, timestamp) AS pansion_last,
          |    max(timestamp) as created,
          |    argMax(is_direction_search, timestamp) AS is_direction_search_int
          |FROM ${clickHouseClient.database}.$table
          |ARRAY JOIN prices AS ps
          |WHERE (from = $from)
          |  AND (has(regions, toUInt32($to)) OR to = $to)
          |  AND (ages = '${ages.sorted.mkString(",")}')
          |  AND ${dateIn("when", dates.when)}
          |  AND ${nightsIn("nights", dates.nights)}
          |  AND ${dates.whenBack.fold("1 = 1")(dateIn("when_back", _))}
          |  AND (timestamp > (now() - ${freshness.toSeconds}))
          |  AND (date >= (today() - ${freshness.toDays}))
          |$whereQuery
          |GROUP BY
          |    hotel_id,
          |    when,
          |    nights,
          |    ps.operator_id,
          |    ps.pansion
          |
          |UNION ALL
          |
          |SELECT hotel_id, when, nights,
          |    when + nights AS when_back,
          |    toInt32(-2), toUInt32(0), toInt8(0),
          |    timestamp as created,
          |    is_direction_search
          |FROM ${clickHouseClient.database}.$table
          |WHERE (from = $from)
          |  AND (has(regions, toUInt32($to)) OR to = $to)
          |  AND (ages = '${ages.sorted.mkString(",")}')
          |  AND ${dateIn("when", dates.when)}
          |  AND ${nightsIn("nights", dates.nights)}
          |  AND ${dates.whenBack.fold("1 = 1")(dateIn("when_back", _))}
          |  AND (timestamp > (now() - ${freshness.toSeconds}))
          |  AND (date >= (today() - ${freshness.toDays}))
          |  AND min_price < 0
      """.stripMargin

    log.debug(s"Executing [$query]")

    for (lines <- clickHouseClient.query(query)) yield {
      val rawPrices = lines.collect {
        case Seq(IntValue(hotelId), when, IntValue(nights), whenBack,
          IntValue(price), IntValue(operator), IntValue(pansion), created, IntValue(isDirectionSearchInt)) =>

          RawPrice(
            raw = Price(
              hotelId,
              LocalDate.parse(when), nights,
              price, operator,
              Pansion.valueOf(pansion),
              dateTimeFormat.parseDateTime(created)
            ),
            isDirectionSearch = isDirectionSearchInt > 0
          )
      }.toVector

      cleanPrices(rawPrices)
    }
  }


  override def searchLastDirectionPrices(from: Int, ages: Seq[Int], dates: SearchDates,
                                         filters: Seq[SnippetFilter],
                                         freshness: FiniteDuration): Future[Seq[DirectionPrice]] = {

    val whereQuery = filters.collect {
      case HotelIdFilter(Some(hotelId)) => s"(hotel_id = $hotelId)"
      case PriceFilter(fromPrice, toPrice) if fromPrice.isDefined || toPrice.isDefined =>
        Seq(
          fromPrice.map(p => "min_price >= " + p),
          toPrice.map(p => "min_price <= " + p)
        ).flatten.mkString(" AND ")
    }.map("  AND " + _).mkString("\n")

    val query =
      s"""SELECT
         |    to,
         |    when,
         |    nights,
         |    when + nights AS when_back,
         |    argMax(min_price, timestamp) AS min_price,
         |    max(timestamp) as created
         |FROM ${clickHouseClient.database}.$table
         |ARRAY JOIN regions as to
         |WHERE (from = $from)
         |  AND (ages = '${ages.sorted.mkString(",")}')
         |  AND ${dateIn("when", dates.when)}
         |  AND ${nightsIn("nights", dates.nights)}
         |  AND ${dates.whenBack.fold("1 = 1")(dateIn("when_back", _))}
         |  AND (timestamp > (now() - ${freshness.toSeconds}))
         |  AND (date >= (today() - ${freshness.toDays}))
         |  AND is_direction_search = 1
         |
         |$whereQuery
         |GROUP BY
         |    to,
         |    when,
         |    nights
         |
         |UNION ALL
         |
         |SELECT
         |    to,
         |    when,
         |    nights,
         |    when_back,
         |    min_price,
         |    created
         |FROM (
         |    SELECT
         |        hotel_id,
         |        argMax(regions, timestamp) AS regions,
         |        when,
         |        nights,
         |        when + nights AS when_back,
         |        argMax(min_price, timestamp) AS min_price,
         |        max(timestamp) as created
         |    FROM ${clickHouseClient.database}.$table
         |    WHERE (from = $from)
         |      AND (ages = '${ages.sorted.mkString(",")}')
         |      AND ${dateIn("when", dates.when)}
         |      AND ${nightsIn("nights", dates.nights)}
         |      AND ${dates.whenBack.fold("1 = 1")(dateIn("when_back", _))}
         |      AND (timestamp > (now() - ${freshness.toSeconds}))
         |      AND (date >= (today() - ${freshness.toDays}))
         |      AND is_direction_search = 0
         |
         |    $whereQuery
         |    GROUP BY
         |        hotel_id,
         |        when,
         |        nights
         |)
         |ARRAY JOIN regions as to
      """.stripMargin

    val q2 =
      s"""SELECT
         |  to,
         |  when,
         |  nights,
         |  argMax(min_price, created) as min_price,
         |  max(created) as created2
         |FROM (
         |${margin(query, 2)}
         |)
         |GROUP BY
         |  to,
         |  when,
         |  nights
         |HAVING
         |  min_price > 0
       """.stripMargin

    val q3 =
      s"""SELECT
         |  to,
         |  argMin(when, min_price) as when,
         |  argMin(nights, min_price) as nights,
         |  min(min_price) as m_price,
         |  argMin(created2, min_price) as price_created
         |FROM (
         |${margin(q2, 2)}
         |)
         |GROUP BY
         |  to
       """.stripMargin

    log.debug(s"Executing [$q3]")

    for (lines <- clickHouseClient.query(q3)) yield {
      lines.collect {
        case Seq(IntValue(to), LocalDateValue(when), IntValue(nights), IntValue(price), created) ⇒

          DirectionPrice(to, when, nights, price, dateTimeFormat.parseDateTime(created))
      }.toVector
    }
  }

  override def getFreshness(from: Int, to: Int, ages: Seq[Int],
                            dates: SearchDates,
                            freshness: FiniteDuration): Future[Map[FlightDay, DateTime]] = {

    val where =
      s"""(from = $from)
         |  AND (to = $to)
         |  AND (ages = '${ages.sorted.mkString(",")}')
         |  AND ${dateIn("when", dates.when)}
         |  AND ${nightsIn("nights", dates.nights)}
         |  AND (timestamp > (now() - ${freshness.toSeconds}))
       """.stripMargin

    val prices =
      s"""SELECT
         |    when,
         |    nights,
         |    max(timestamp) AS price_timestamp
         |FROM ${clickHouseClient.database}.tour_prices
         |WHERE $where
         |GROUP BY when, nights
       """.stripMargin

    val empty =
      s"""SELECT
         |    when,
         |    nights,
         |    max(timestamp) AS empty_timestamp
         |FROM ${clickHouseClient.database}.tour_empty_results
         |WHERE $where
         |GROUP BY when, nights
       """.stripMargin

    val query =
      s"""SELECT
         |  when, nights,
         |  (price_timestamp > empty_timestamp) ? price_timestamp : empty_timestamp
         |FROM (
         |  $prices
         |)
         |ANY LEFT JOIN (
         |  $empty
         |) USING when, nights
       """.stripMargin

    for (lines <- clickHouseClient.query(query)) yield {
      lines.collect {
        case Seq(LocalDateValue(when), IntValue(nights), created) =>
          FlightDay(when, nights) -> dateTimeFormat.parseDateTime(created)
      }.toMap
    }
  }
}
