package ru.yandex.tours.resorts

import play.api.libs.json._
import play.api.libs.json.extensions._
import ru.yandex.tours.model.Image
import ru.yandex.tours.model.Image.Size
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.model.image.ImageProviders.ImageProvider
import ru.yandex.tours.resorts.VoteTypes.VoteType
import ru.yandex.tours.serialize.json.CommonJsonFormats._
import java.time.{Month, Year}

/**
 * Created by Evgeny Zhoga on 02.12.15.
 */
case class SkiInput(id: Int,
                    parentResortId: Option[Int] = None,
                    subregionsIds: Option[List[Int]] = None,
                    name: String,
                    geoIds: List[Int],
                    isMetaRegion: Boolean,
                    url: Option[String],
                    partnersUrls: List[String],
                    description: String,
                    seasonStartMonth: Option[Month] = None,
                    seasonEndMonth: Option[Month] = None,
                    slopes: Option[Slopes],
                    lifts: Option[Lifts],
                    features: Option[List[String]],
                    skiMapUrls: Option[List[Resource]] = None,
                    imagesUrls: Option[List[Resource]] = None,
                    skiPassInfo: Option[SkiPassInfo] = None,
                    reviews: Option[List[Review]],
                    skiRuUrl: Option[String],
                    nearDistance: Option[Int]) {

  private def seasonBounds: Option[SeasonBounds] =
    if (seasonStartMonth.isDefined || seasonEndMonth.isDefined) Some(SeasonBounds(seasonStartMonth, seasonEndMonth))
    else None

  val asSki = Ski(
    id, parentResortId, subregionsIds,
    None, name, geoIds.map(Region(_, None, None, None, None, None)),
    isMetaRegion, url, partnersUrls, description,
    seasonBounds,
    slopes, lifts, features, skiMapUrls, None, skiPassInfo, None, None,
    skiRuUrl, nearDistance
  )
}
case class Ski(id: Int,
               parentResortId: Option[Int] = None,
               subregionsIds: Option[List[Int]] = None,
               subregions: Option[List[Region]] = None,
               name: String,
               regions: List[Region],
               isMetaRegion: Boolean,
               url: Option[String],
               partnersUrls: List[String],
               description: String,
               seasonBounds: Option[SeasonBounds],
               slopes: Option[Slopes],
               lifts: Option[Lifts],
               features: Option[List[String]],
               skiMapUrls: Option[List[Resource]] = None,
               images: Option[List[ImageResource]] = None,
               skiPassInfo: Option[SkiPassInfo] = None,
               mainImage: Option[ImageResource],
               location: Option[Location] = None,
               skiRuUrl: Option[String],
               nearDistance: Option[Int]
              // please note, that 22 parameters is maximum, allowed by JVM!!!
              )

case class Resort(ski: Ski, reviewsInformation: Option[ReviewsInformation])

case class Location(region: Option[Region], country: Option[Region], coordinates: Option[Coordinates])

case class Slopes(length: Option[Int] = None,
                  longest: Option[Int] = None,
                  height: Int,
                  minHeight: Int,
                  slopesByType: List[Slope])

case class Slope(`type`: SlopeType.SlopeType,
                 length: Option[Int] = None,
                 count: Option[Int] = None)

object SlopeType extends Enumeration {
  type SlopeType = Value
  val GREEN, BLUE, RED, BLACK, FREERIDE, WITH_LIGHT, CROSS_COUNTRY_SKI = Value
}

case class SeasonBounds(startMonth: Option[Month], endMonth: Option[Month])

case class Lifts(liftStartAt: Option[String] = None,
                 liftEndAt: Option[String] = None,
                 liftsByType: List[Lift])

case class Lift(`type`: LiftType.LiftType, count: Int)

object LiftType extends Enumeration {
  type LiftType = Value
  val BELT, BUGEL, CHAIR, CABIN = Value
}

case class Resource(url: String, owner: String)

case class ImageResource(owner: String,
                         urlBase: String)

case class SkiPassInfo(skiPassSource: Option[String], skiPassRates: Option[List[SkiPass]])

case class SkiPass(title: Option[String] = None,
                   comment: Option[String] = None,
                   freeOfCharge: Option[Int] = None,
                   periods: List[Period],
                   rates: List[Rate])

case class Period(from: String, to: String)

case class Rate(precision: Precision.Precision,
                count: Int,
                period: Option[Int] = None,
                comment: Option[String] = None,
                age: Age,
                price: Price)

object Precision extends Enumeration {
  type Precision = Value
  val HOUR, DAY, SEASON, HOLIDAY, HALFDAY = Value
}

case class Age(name: AgeName.AgeName,
               ageFrom: Option[Int] = None,
               ageTo: Option[Int])

object AgeName extends Enumeration {
  type AgeName = Value

  val `ДЕТСКИЙ`, `ШКОЛЬНЫЙ`, `МОЛОДЁЖНЫЙ`, `ВЗРОСЛЫЙ`, `ПЕНСИОННЫЙ`, `СТУДЕНЧЕСКИЙ` = Value
}

case class Price(amount: BigDecimal, currency: Currency.Currency) {
  def /(div: Int) = copy(amount = amount / div)
}

object Currency extends Enumeration {
  type Currency = Value
  val `РУБ`, EUR, USD, SEK, CHF, `$`, TL, GEL, AMD, BGN, `ГРН`, PLN = Value
}

case class Review(id: Int,
                  userName: String,
                  date: String,
                  stayYear: Option[Year],
                  stayMonth: Option[Month],
                  link: Option[String],
                  reviewText: Option[String],
                  votes: List[Vote])

case class Region(id: Int,
                  name: Option[String],
                  genitive: Option[String],
                  accusative: Option[String],
                  preposition: Option[String],
                  prepositional: Option[String])

case class Coordinates(latSpan: Double,
                       latitude: Double,
                       lonSpan: Double,
                       longitude: Double)

object VoteTypes extends Enumeration {
  type VoteType = Value

  val ACCOMMODATION, APRESSKI, LIFTS, PEOPLE, PRICE, SKIING, WEATHER = Value

}

case class Vote(`type`: VoteType,
                rating: Option[Int],
                comment: Option[String])

case class ReviewsInformation(resortId: Int,
                              resortStats: ReviewsInformation.ResortStats,
                              reviews: List[ReviewsInformation.EReview])

object ReviewsInformation {
  case class Author(signPrivacy: String, name: String)
  case class Link(rel: String, href: String)

  case class EReview(id: String,
                     dtreviewed: Long,
                     resortId: Int,
                     url: Option[String],
                     authors: List[Author],
                     snippet: Option[String],
                     stayYear: Option[Year],
                     stayMonth: Option[Month],
                     attitudes: List[Attitude],
                     dataProvider: DataProvider)

  case class Attitude(name: VoteType, score: Option[Double], description: Option[String])
  case class DataProvider(name: String, url: String)

  case class ResortStats(resortId: Int,
                         reviewsCount: Int,
                         scoresCount: Int,
                         score: Option[Double],
                         attitudes: List[Attitude])

  implicit val _riDataProviderJsonFormat = Json.format[ReviewsInformation.DataProvider]
  implicit val _internalVoteTypeEnumFormat = Implicits._voteTypeEnumFormat
  implicit val _riAttitdeJsonFormat = Json.format[ReviewsInformation.Attitude]
  implicit val _riAuthorJsonFormat = Json.format[ReviewsInformation.Author]
  implicit val _riLinkJsonFormat = Json.format[ReviewsInformation.Link]
  implicit val _riReviewJsonFormat = Json.format[ReviewsInformation.EReview]
  implicit val _riResortStatsJsonFormat = Json.format[ReviewsInformation.ResortStats]
}


object Implicits {
  // http://stackoverflow.com/questions/23816753/play-framework-how-to-serialize-deserialize-an-enumeration-to-from-json
  @scala.annotation.tailrec
  def fix(s: String, toLower: Boolean = true): String =
    if (s.startsWith("\"")) fix(s.stripPrefix("\""), toLower)
    else if (s.endsWith("\"")) fix(s.stripSuffix("\""), toLower)
    else
      if (toLower) s.toLowerCase else s

  implicit val _slopeTypeEnumFormat = enumeration(SlopeType.values)

  implicit val _liftTypeEnumFormat = enumeration(LiftType.values)

  implicit val _precisionEnumFormat = enumeration(Precision.values)

  implicit val _ageNameEnumFormat = enumeration(AgeName.values)

  implicit val _currencyEnumFormat = enumeration(Currency.values)

  implicit val _imageProviderEnumFormat = enumeration(ImageProviders.values)

  implicit val _voteTypeEnumFormat = enumeration(VoteTypes.values)

  /* formats are used for case class conversions to/from json. Not supposed to be used explicitly */
  implicit val _voteJsonFormat = Json.format[Vote]
  implicit val _reviewJsonFormat = Json.format[Review]

  import ReviewsInformation._

  implicit val _reviewsInformationJsonFormat = Json.format[ReviewsInformation]

  implicit val _ageJsonFormat = Json.format[Age]
  implicit val _liftJsonFormat = Json.format[Lift]
  implicit val _liftsJsonFormat = Json.format[Lifts]
  implicit val _priceJsonFormat = Json.format[Price]
  implicit val _rateJsonFormat = Json.format[Rate]
  implicit val _resourceJsonFormat = Json.format[Resource]
  implicit val _periodJsonFormat = Json.format[Period]
  implicit val _skiPassJsonFormat = Json.format[SkiPass]
  implicit val _skiPassInfoJsonFormat = Json.format[SkiPassInfo]
  implicit val _slopeJsonFormat = Json.format[Slope]
  implicit val _slopesJsonFormat = Json.format[Slopes]

  implicit val _imageResourceFormat = Json.format[ImageResource]
  implicit val _skiInputJsonFormat = Json.format[SkiInput]
  implicit val _regionJsonFormat = Json.format[Region]
  implicit val _seasonBoundsJsonFormat = Json.format[SeasonBounds]
  implicit val _coordinatesJsonFormat = Json.format[Coordinates]
  implicit val _locationJsonFormat = Json.format[Location]
  implicit val _skiJsonFormat = Json.format[Ski]

  implicit val _resortJsonFormat = Json.format[Resort]

  // http://stackoverflow.com/questions/28956243/play-2-x-json-transform-json-keys-to-camelcase-from-underscore-case
  private def underscoreToCamel(name: String) = "_([a-z\\d])".r.replaceAllIn(name, {m =>
    m.group(1).toUpperCase
  })
  private def camelToUnderscore(name: String) = "([a-z\\d])([A-Z])".r.replaceAllIn(name, {m =>
    m.group(1) + "_" + m.group(2).toLowerCase
  })
  private def camelToDash(name: String) = "([a-z\\d])([A-Z])".r.replaceAllIn(name, {m =>
    m.group(1) + "-" + m.group(2).toLowerCase
  })
  def underscoreToCamelCaseJs(json: JsValue): JsValue = json.updateAllKeyNodes {
    case (path, js) => JsPathExtension.hasKey(path) match {
      case Some(key) => underscoreToCamel(key) -> js
      case None => path.toJsonString -> js
    }
  }
  def camelCaseToUnderscoreJs(json: JsValue): JsValue = json.updateAllKeyNodes {
    case (path, js) => JsPathExtension.hasKey(path) match {
      case Some(key) => camelToUnderscore(key) -> js
      case None => path.toJsonString -> js
    }
  }
  def camelCaseToDashJs(json: JsValue): JsValue = json.updateAllKeyNodes {
    case (path, js) => JsPathExtension.hasKey(path) match {
      case Some(key) => camelToDash(key) -> js
      case None => path.toJsonString -> js
    }
  }
}