package ru.yandex.tours.resorts

import java.io.InputStream

import play.api.libs.json.{JsError, JsSuccess, Json}
import ru.yandex.extdata.common.exception.{DataIsNotLoadedException, UnableToLoadDataException}
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.extdata.{DataDef, DataTypes}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.Logging

import scala.io.Source

/**
 * Created by Evgeny Zhoga on 04.12.15.
 */
class SkiResorts(val resorts: List[Resort]) {
  private val byId = resorts.map(r => r.ski.id -> r).toMap
  private val byRegion = resorts.flatMap(r => r.ski.regions.map(id => id.id -> r)).toMultiMap
    .map((sortByMatching _).tupled)

  def get(id: Int): Option[Resort] = byId.get(id)
  def getByRegion(regionId: Int): List[Resort] = byRegion.getOrElse(regionId, List.empty)

  private def sortByMatching(id: Int, resorts: List[Resort]): (Int, List[Resort]) = {
    val (exact, rest) = resorts.partition(_.ski.id == id)
    val sorted = exact ++ rest
    id -> sorted
  }
}

object SkiResorts extends DataDef[SkiResorts] with Logging {
  override def dataType: DataType = DataTypes.skiResorts

  override def from(extDataService: ExtDataService): SkiResorts = {
    try {
      super.from(extDataService)
    } catch {
      case _: DataIsNotLoadedException | _: UnableToLoadDataException =>
        log.warn(s"No previous ${dataType.getName}. Using empty one.")
        empty
    }
  }
  override def parse(is: InputStream): SkiResorts = {
    import ru.yandex.tours.resorts.Implicits._

    val content = Source.fromInputStream(is, "UTF-8").mkString
    val json = underscoreToCamelCaseJs(Json.parse(content)).validate[List[Resort]]

    json match {
      case JsSuccess(listOfResorts, _) =>
        val patchedResorts = listOfResorts.map {
          case resort@Resort(ski, ri) =>
            // divide all ski pass rates by 100
            val skiPassInfo = ski.skiPassInfo.map { spi =>
              val rates = spi.skiPassRates.map { skiPassRates =>
                skiPassRates.map {
                  skiPass => skiPass.copy(rates = skiPass.rates.map(r => r.copy(price = r.price / 100)))
                }
              }
              spi.copy(skiPassRates = rates)
            }

            resort.copy(ski = ski.copy(
              skiPassInfo = skiPassInfo
            ))
        }

        new SkiResorts(patchedResorts)
      case JsError(e) =>
        throw new RuntimeException(s"Ski resorts parsing error: $e")
    }

  }

  def empty: SkiResorts = new SkiResorts(List.empty[Resort])
}
