package ru.yandex.tours.search.settings

import play.api.libs.json.Json
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.settings.{Parameters, SearchSettingsRecord}
import ru.yandex.tours.model.search.{SearchOptions, SearchType}

/**
 * regionId: region id
 * banned: if region is not searchable because it or any of it's parent is banned
 * allowed: if region itself is marked as allowed for search or allowed by default. Take care about banned as well!
 * disallowed: if region itself is marked as allowed for search or allowed by default.
 * regionModified: tells if current region is differ from default
 */
case class RegionSearchSettings(regionId: Option[Int],
                                banned: Map[SearchType, Boolean],
                                allowed: Map[SearchType, Boolean],
                                disallowed: Map[SearchType, Boolean],
                                regionModified: Map[SearchType, Boolean],
                                forceSearch: Boolean,
                                ltActive: Option[Boolean]) {

  def isAllowed(searchType: SearchType): Boolean = allowed(searchType) && !banned(searchType)
  def isSearchable: Boolean = (forceSearch || allowed.exists(_._2 == true)) && banned.exists(_._2 == false)
  val forceSearchIfUnknown = forceSearch
}


object RegionSearchSettings {
  object Implicits {
    import ru.yandex.tours.serialize.json.CommonJsonFormats._

    implicit val searchOptionsToJson = enumeration(SearchOptions.values)
    implicit val searchTypesToJson = enumeration(SearchType.values)
    implicit val mapToJson = mapFormat(s => SearchType.withName(s), s => SearchOptions.withName(s))
    implicit val parametersToJson = Json.format[Parameters]
    implicit val searchSettingsRecordToJson = Json.format[SearchSettingsRecord]
  }
  def apply(region: Region,
            parameters: Map[(Int, SearchType), SearchOptions.SearchOption],
            forceSearch: Boolean, ltActive: Option[Boolean] = None): RegionSearchSettings = {

    val parametersForRegion = parameters.filter(_._1._1 == region.id).map {case ((_, st), so) => st -> so}
    val regionModified = SearchType.values
      .map { searchType => searchType -> parametersForRegion.contains(searchType) }
      .toMap
    val allowed = SearchType.values
      .map { searchType => searchType -> (parametersForRegion.getOrElse(searchType, default(searchType, region)) == SearchOptions.OPEN) }
      .toMap
    val disallowed = SearchType.values
      .map { searchType => searchType -> (parametersForRegion.getOrElse(searchType, default(searchType, region)) == SearchOptions.CLOSED) }
      .toMap

    val banned = SearchType.values.
      map {searchType => searchType -> parameters.exists {
        case ((_, st), so) if st == searchType && so == SearchOptions.BANNED => true
        case _ => false
      }}.toMap

    RegionSearchSettings(Some(region.id), banned, allowed, disallowed, regionModified, forceSearch, ltActive)
  }

  private val closedTypes = Set(
    Types.Airport,
    Types.CityDistrict,
    Types.SecondLevelCityDistrict,
    Types.MetroStation,
    Types.MonorailStation
  )

  private def default(searchType: SearchType, region: Region) =
    if (region.isCountry && searchType == SearchType.ROOMS) SearchOptions.CLOSED
    else if (closedTypes.contains(region.`type`)) SearchOptions.CLOSED
    else SearchOptions.OPEN

  /**
    * Default answer - banned, notAllowed, not modified
    */
  val UnknownRegionSearchSettings =
    RegionSearchSettings(
      None,
      SearchType.values.map(v => v -> true).toMap,
      SearchType.values.map(v => v -> false).toMap,
      SearchType.values.map(v => v -> true).toMap,
      SearchType.values.map(v => v -> false).toMap,
      forceSearch = false,
      ltActive = None)
}