package ru.yandex.tours.search.settings

import java.io.InputStream

import play.api.libs.json.{JsSuccess, Json}
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDefWithDependencies, DataTypes}
import ru.yandex.tours.geo.LtActiveCountries
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.search.BaseRequest
import ru.yandex.tours.model.search.settings.SearchSettingsRecord
import shapeless.{::, HNil}

import scala.io.Source

/**
 * mappings regionId -> searchAllowed
 * Created by Evgeny Zhoga on 01.02.16.
 */
class SearchSettingsHolder(settings: Seq[SearchSettingsRecord], tree: Tree) {

  def getRegionSearchSettings(region: Region): RegionSearchSettings = {
    val path = tree.pathToRoot(region)
    val idsPath = path.map(_.id).toSet
    val res = settings.filter(ssr => idsPath.contains(ssr.regionId))

    val params = res.flatMap { sc =>
      sc.parameters.searchByType.toSeq.map {
        case (searchType, searchOption) => (sc.regionId, searchType) -> searchOption
      }
    }.toMap
    val forcesSearch = res.find(_.regionId == region.id).flatMap(_.parameters.forcedSearchable).getOrElse(false)
    val ltActive = res.flatMap(_.parameters.ltActive).reverse.headOption

    RegionSearchSettings(region, params, forcesSearch, ltActive)
  }

  def allForcedRegionIds: Set[Int] = {
    settings.filter(_.parameters.forcedSearchable.contains(true)).map(_.regionId).toSet
  }

  def getRegionSearchSettings(regionId: Int): RegionSearchSettings =
    tree.region(regionId).map(getRegionSearchSettings).
      getOrElse(RegionSearchSettings.UnknownRegionSearchSettings)

  def isTooClose(request: BaseRequest): Boolean = {
    isTooClose(request.hotelRequest.from, request.hotelRequest.to)
  }

  def isTooClose(from: Int, to: Int): Boolean = {
    tree.findParents(from).exists(r => !r.isCountry && r.id == to) ||
      tree.findParents(to).exists(_.id == from)
  }
}

object SearchSettingsHolder extends DataDefWithDependencies[SearchSettingsHolder, Tree :: LtActiveCountries :: HNil] {
  override def dataType: DataType = DataTypes.searchSettings
  override def dependsOn: Set[DataType] = Set(DataTypes.regions, DataTypes.ltActiveCountries)

  def empty: SearchSettingsHolder = new SearchSettingsHolder(Seq.empty, Tree.empty)

  override def parse(is: InputStream, dependencies: Tree :: LtActiveCountries :: HNil): SearchSettingsHolder = {
    import RegionSearchSettings.Implicits._

    val tree :: ltActiveCountries :: HNil = dependencies

    val mapping = Json.parse(Source.fromInputStream(is).mkString).as[Seq[SearchSettingsRecord]]

    def mappingWithLt = mapping.map { record =>
      if (ltActiveCountries.set.contains(record.regionId))
        record.setLtActive()
      else record
    }

    def ltRecords = (ltActiveCountries.set -- mapping.map(_.regionId)).toSeq.map { id =>
      SearchSettingsRecord.empty(id).setLtActive()
    }

    new SearchSettingsHolder(mappingWithLt ++ ltRecords, tree)
  }
}
