package ru.yandex.tours.serialize

import java.net.URLEncoder

import org.joda.time.LocalDate
import org.joda.time.format.DateTimeFormat
import ru.yandex.tours.model.direction.Thematics.Thematic
import ru.yandex.tours.model.filter._
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.util.SortType.SortType
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.search.DefaultRequestGenerator
import ru.yandex.tours.util.http.UrlSafe

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 23.01.15
 */
class UrlBuilder(val domain: String) {

  private val format = DateTimeFormat.forPattern("yyyy-MM-dd")
  
  def forHome(context: SearchType, utm: UtmMark = UtmMark.empty): String = {
    s"//$domain/?context=${context.toString.toLowerCase}&${utm.urlSafe}"
  }

  def forThematic(thematic: Thematic, context: SearchType,
                  filters: Map[String, String] = Map.empty,
                  utm: UtmMark = UtmMark.empty): String = {
    val filtersSafe = UrlSafe.paramsFromMap(filters)
    s"//$domain/${thematic.toString.toLowerCase}?context=${context.toString.toLowerCase}&$filtersSafe&${utm.urlSafe}"
  }

  def forSkiDirection(direction: Int): String = {
    s"//$domain/resort/ski/$direction"
  }

  val faviconDomain = "travel.yandex.ru"
  private val ages = DefaultRequestGenerator.DEFAULT_AGES.map("ages=" + _).mkString("&")
  private def escape(x: String) = URLEncoder.encode(x, "UTF-8")

  def forHotel(from: Int, hotelId: Int, when: LocalDate, nights: Int, operatorId: Option[Int] = None, utm: UtmMark = UtmMark.empty, context: SearchType) = {
    val operatorFilter = operatorId.fold("")("&tour_operator=" + _)
    forHotelDefault(hotelId, utm, context) + s"&from=$from&when_flex=false&nights_flex=false&when=${when.toString(format)}&nights=$nights&$ages$operatorFilter"
  }

  def forHotelDefault(hotelId: Int, utm: UtmMark, context: SearchType) = {
    s"//$domain/hotel/$hotelId?context=${context.toString.toLowerCase}&${utm.urlSafe}"
  }

  def forHotelReviews(from: Int, hotelId: Int, when: LocalDate, nights: Int, utm: UtmMark = UtmMark.empty) = {
    s"//$domain/hotel/$hotelId/reviews?from=$from&when_flex=false&nights_flex=false&when=${when.toString(format)}&nights=$nights&$ages&${utm.urlSafe}"
  }

  def forHotelPhoto(from: Int, hotelId: Int, when: LocalDate, nights: Int, photoId: String, operatorId: Option[Int] = None, utm: UtmMark = UtmMark.empty) = {
    val operatorFilter = operatorId.fold("")("&tour_operator=" + _)
    s"//$domain/hotel/$hotelId?from=$from&when_flex=false&nights_flex=false&when=${when.toString(format)}&nights=$nights&$ages$operatorFilter&photo_id=$photoId&${utm.urlSafe}"
  }

  def forHotelAnchor(from: Int, hotelId: Int, when: LocalDate, nights: Int, anchor: String, operatorId: Option[Int] = None, utm: UtmMark = UtmMark.empty) = {
    val operatorFilter = operatorId.fold("")("&tour_operator=" + _)
    s"//$domain/hotel/$hotelId?from=$from&when_flex=false&nights_flex=false&when=${when.toString(format)}&nights=$nights&$ages$operatorFilter&${utm.urlSafe}#$anchor"
  }

  def forSearch(from: Int, to: Int, when: LocalDate, nights: Int,
                operatorId: Option[Int] = None,
                filters: Seq[HotelFilter] = Seq.empty,
                sortType: Option[SortType] = None,
                utm: UtmMark = UtmMark.empty, context: SearchType) = {

    val base =
      if (context == SearchType.TOURS) s"//$domain/search"
      else s"//$domain/search/rooms"

    val sb = new StringBuilder
    sb.append(base)
      .append("?from=").append(from)
      .append("&to=").append(to)
      .append("&when_flex=false&nights_flex=false")
      .append("&when=").append(when.toString(format))
      .append("&nights=").append(nights)
      .append("&").append(ages)

    operatorId.foreach { id =>
      sb.append("&tour_operator=").append(id)
    }
    sortType.foreach { sort =>
      sb.append("&sort=").append(sort.toString.toLowerCase)
    }
    filters.foreach { filter =>
      filter.values.foreach {
        case StringValue(x) => sb.append("&").append(filter.name).append("=").append(escape(x))
        case IntValue(x) => sb.append("&").append(filter.name).append("=").append(x)
        case BooleanValue(x) => sb.append("&").append(filter.name).append("=").append(x)
        case DoubleValue(x) => sb.append("&").append(filter.name).append("=").append(x)
        case NullValue =>
      }
    }

    sb.append("&").append(utm.urlSafe)

    sb.toString()
  }
}
