package ru.yandex.tours.serialize.json

import java.time.DateTimeException

import org.joda.time.{DateTimeZone, LocalDate}
import play.api.libs.json._
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.model.{Languages, LocalizedString, MapRectangle}

/**
 * Json formats for common model objects
 */
trait CommonJsonFormats {

  def enumeration[T <: Enumeration#Value](values: Iterable[T]): Format[T] = Format[T](
    Reads { js =>
      val toFind = js.as[String]
      val found = values.find(_.toString.equalsIgnoreCase(toFind))
        .getOrElse(sys.error(s"Not found value $js in $values"))
      JsSuccess(found)
    },
    Writes(e => JsString(e.toString))
  )

  implicit val localizedStringFormat = Format[LocalizedString](
    Reads { js =>
      val map = js.as[JsObject].value.toSeq.flatMap {
        case (lang, JsString(value)) => Some(Languages.withName(lang) -> value)
        case _ => None
      }
      JsSuccess(LocalizedString(map: _*))
    },
    Writes { str =>
      str.values.foldLeft(Json.obj()) {
        case (js, (lang, value)) => js + (lang.toString -> JsString(value))
      }
    }
  )
  def mapFormat[K, V](stringToK: String => K, stringToV: String => V): Format[Map[K, V]] = Format[Map[K, V]](
    Reads {js =>
      val json = js.as[JsObject]
      JsSuccess(json.keys.map(k => stringToK(k) -> stringToV((json \ k).as[String])).toMap)
    },
    Writes { map =>
      map.foldLeft(Json.obj()){case (acc, (key, value)) => acc + (key.toString -> JsString(value.toString))}
    }
  )

  implicit val regionTypeFormat = enumeration[Types.Value](Types.values)
  implicit val localDateFormat = Format[LocalDate](
    Reads(js => JsSuccess(LocalDate.parse(js.as[String]))),
    Writes(ld => JsString(ld.toString))
  )
  implicit val dateTimeZoneFormat = Format[DateTimeZone](
    Reads(js => JsSuccess(DateTimeZone.forID(js.as[String]))),
    Writes(dtz => JsString(dtz.toString))
  )

  implicit val mapRectFormat = Json.format[MapRectangle]

  implicit val monthJsonFormat = Format[java.time.Month] (
    Reads { js =>
      try JsSuccess(java.time.Month.of(js.as[Int]))
      catch { case e: DateTimeException => sys.error(s"Wrong month: $js") }
    },
    Writes(month => JsNumber(month.getValue))
  )
  implicit val yearJsonFormat = Format[java.time.Year] (
    Reads { js =>
      try JsSuccess(java.time.Year.of(js.as[Int]))
      catch { case e: DateTimeException => sys.error(s"Wrong year: $js") }
    },
    Writes(year => JsNumber(year.getValue))
  )
}

object CommonJsonFormats extends CommonJsonFormats