package ru.yandex.tours.serialize.json

import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.agencies.AgenciesIndex.{AgenciesInRegion, AgencyWithBilling}
import ru.yandex.tours.avia.AviaAirportRecommendations.AirportRecommendation
import ru.yandex.tours.billing.BillingOffers
import ru.yandex.tours.direction.Direction
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.{Tree, Types}
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.{HotelsPanoramas, HotelsVideo}
import ru.yandex.tours.metro.{MetroHolder, MetroUtil}
import ru.yandex.tours.model.Agencies.Metro
import ru.yandex.tours.model.filter._
import ru.yandex.tours.model.hotels.Features._
import ru.yandex.tours.model.hotels.{Address, Hotel}
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.model.search.{GetOfferRequest, HotelSearchRequest, OfferSearchRequest, SearchFilter}
import ru.yandex.tours.model.util.SortType.SortType
import ru.yandex.tours.model.{Agency, Image, LocalizedString, MapRectangle}
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.naming.HotelNameUtils
import ru.yandex.tours.model.Languages._

import scala.collection.JavaConversions._

trait CommonJsonSerialization {

  def filtersToJson(filters: Iterable[Filter]): JSONObject = {
    val ans = new JSONObject()
    for (filter <- filters) {
      val name = filter.name
      val values = filter.values
      if (values.contains(BooleanValue(x = true))) {
        ans.put(name, true)
      } else {
        if (values.nonEmpty) {
          val array = new JSONArray()
          values.foreach {
            case StringValue(x) => array.put(x)
            case IntValue(x) => array.put(x)
            case BooleanValue(x) => array.put(x)
            case DoubleValue(x) => array.put(x)
            case NullValue => array.put(null: Object)
          }
          ans.put(name, array)
        }
      }
    }
    ans
  }

  def toJson(request: OfferSearchRequest, tree: Tree, lang: Lang): JSONObject = {
    val ans = toJson(request.hotelRequest, tree, lang, None)
    ans.getJSONObject("to").put("hotel_id", request.hotelId)
    ans
  }

  def toJson(request: GetOfferRequest, tree: Tree, lang: Lang): JSONObject = {
    toJson(request.toursInHotelRequest, tree, lang)
      .put("tour_id", request.tourId)
  }

  def toJson(searchRequest: HotelSearchRequest, tree: Tree, lang: Lang, sortBy: Option[SortType]): JSONObject = {
    val ans = new JSONObject()
    ans.put("from", searchRequest.from)
    ans.put("from_info", regionToJson(searchRequest.from, tree, lang))
    ans.put("to", regionToJson(searchRequest.to, tree, lang))
    ans.put("nights", searchRequest.nights)
    ans.put("when", searchRequest.when)
    val ages = new JSONArray()
    searchRequest.ages.foreach(ages.put)
    ans.put("ages", ages)
    ans.put("when_flex", searchRequest.flexWhen)
    ans.put("nights_flex", searchRequest.flexNights)
    ans.put("currency", searchRequest.currency.name())
    ans.put("lang", searchRequest.lang.toString)
    Some(searchRequest.filter).map(_.toString).filterNot(_ == "")
      .foreach { ans.put(SearchFilter.name, _) }
    sortBy.foreach(s => ans.put("sort", s.toString.toLowerCase))
    ans
  }

  def toJson(features: Iterable[FeatureValue]): JSONObject = {
    val ans = new JSONObject()
    features.foreach({
      case IntFeatureValue(feature, optValue) => optValue.map(value => ans.put(feature.name, value))
      case EnumFeatureValue(feature, optValue) =>
        optValue.foreach(value => ans.put(feature.name, value))
      case MultipleFeatureValue(feature, value) =>
        if (value.nonEmpty) {
          val ar = new JSONArray()
          value.foreach(ar.put)
          ans.put(feature.name, ar)
        }
      case BooleanFeatureValue(feature, optValue) =>
        optValue.foreach(value => ans.put(feature.name, value))
    })
    ans
  }

  def toJson(photo: Image): JSONObject = {
    val ans = new JSONObject()
    ans.put("id", photo.name)
    ans.put("base_url", photo.baseUrlWithoutProtocol)
    ans.put("provider", photo.provider)
    if (ImageProviders.isChannelManager(photo.provider)) {
      ans.put("is_cmhotel", true)
    }
    ans
  }

  def toJson(recommendation: AirportRecommendation, tree: Tree, lang: Lang): JSONObject = {
    val ans = new JSONObject()

    val name = recommendation.city.geoId.flatMap(tree.region).map(_.name(lang))
      .getOrElse(recommendation.city.name(lang))

    ans.put("id", recommendation.city.id)
    ans.put("name", name)
    recommendation.distance.foreach(ans.put("distance", _))
    ans
  }

  def breadcrumbsToJson(geoId: Int, tree: Tree,
                        geoMapping: GeoMappingHolder,
                        searchSettings: SearchSettingsHolder,
                        lang: Lang): JSONArray = {
    val regions = new JSONArray()
    def isGoodRegion(x: Region): Boolean = {
      geoMapping.isKnownDestination(x.id) && searchSettings.getRegionSearchSettings(x.id).isSearchable
    }
    tree.region(geoId).foreach(region => {
      tree.pathToRoot(region).filter(isGoodRegion).reverse.foreach { region =>
        regions.put(regionToJson(region, lang))
      }
      if (regions.length() == 0) {
        regions.put(regionToJson(region, lang))
      }
    })
    regions
  }

  def toJson(hotel: Hotel, hotelsPanoramas: HotelsPanoramas): Option[JSONObject] = {
    hotelsPanoramas.getPanorama(hotel).map { panorama ⇒
      new JSONObject()
        .put("preview", toJson(panorama.preview))
        .put("url", panorama.mapsUrl)
    }
  }

  def toJson(hotel: Hotel,
             hotelsVideo: HotelsVideo,
             tree: Tree,
             geoMapping: GeoMappingHolder,
             searchSettings: SearchSettingsHolder,
             lang: Lang): JSONObject = {
    val address = hotel.addresses.getOrElse(lang,
      if (hotel.addresses.isEmpty) Address.empty(lang) else hotel.addresses.values.head)

    val name = hotel.name(lang)
    val ans = new JSONObject()
      .put("id", hotel.id)
      .put("geo_id", hotel.geoId)
      .put("name", name)
      .put("skip_prefix", HotelNameUtils.hasHotelPrefix(name))
      .put("country", address.country)
      .put("city", address.locality)
      .put("address", address.fullAddress)
      .put("stars", hotel.star.id)
      .put("features", toJson(hotel.features))
      .put("rating", hotel.rating)
      .put("reviews_count", hotel.reviewsCount)
      .put("longitude", hotel.longitude)
      .put("latitude", hotel.latitude)
      .put("rooms_search", hotel.roomsSearchAvailable)
      .put("tours_search", hotel.toursSearchAvailable)
      .put("to", regionToJson(hotel.geoId, tree, lang))
      .put("backa_permalink", hotel.backaPermalink.getOrElse(JSONObject.NULL))
    ans.put("regions", breadcrumbsToJson(hotel.geoId, tree, geoMapping, searchSettings, lang))
    tree.region(hotel.geoId).foreach { region =>
      tree.country(region).foreach(country => ans.put("country", country.name(lang)))
      tree.parent(region, Types.City).foreach(city => ans.put("city", city.name(lang)))
    }

    val images = new JSONArray()
    hotel.images.foreach(t => images.put(toJson(t)))
    ans.put("images", images)
    hotel.images.headOption.foreach { t =>
      ans.put("bad_photo", t.badPhoto)
    }

    for (video <- hotelsVideo.getVideo(hotel)) {
      ans.put("video", new JSONObject().put("preview", toJson(video.preview)).put("url", video.videoUrl))
    }

    ans
  }

  def regionToJson(geoId: Int, tree: Tree, lang: Lang): JSONObject = {
    val ans = new JSONObject().put("id", geoId)
    tree.region(geoId).fold(ans)(region => regionToJson(region, lang))
  }

  protected def regionName(id: Int, tree: Tree, lang: Lang) = {
    tree.region(id).map(_.name(lang)).getOrElse("Unknown")
  }

  def directionToJson(direction: Direction, lang: Lang): JSONObject = {
    regionToJson(direction.region, lang)
      .put("photo", toJson(direction.mainImage)) // for email renderer
      .put("image", toJson(direction.mainImage)) // for travel.yandex.ru verstka
  }

  def regionToJson(region: Region, lang: Lang): JSONObject = {
    new JSONObject()
      .put("id", region.id)
      .put("name", region.name(lang))
      .put("genitive", region.genitive)
      .put("accusative", region.accusative)
      .put("preposition", region.preposition)
      .put("prepositional", region.locative)
  }

  def agenciesToJson(agencies: Iterable[AgencyWithBilling], metroHolder: MetroHolder, lang: Lang,
                     metroColor: Option[String] = None): JSONArray = {
    val ar = new JSONArray()
    agencies.foreach(agency => ar.put(agencyToJson(agency, metroHolder, lang, metroColor)))
    ar
  }

  def agencyToJson(agency: Agency, metroHolder: MetroHolder, lang: Lang, metroColor: Option[String]): JSONObject = {
    val phones = new JSONArray()
    agency.phones.foreach(phones.put)
    val optState = agency.currentWorkTimeStatus
    val ans = new JSONObject()
      .put("id", agency.id)
      .put("geo_id", agency.geoId)
      .put("name", agency.names(lang))
      .put("longitude", agency.longitude)
      .put("latitude", agency.latitude)
      .put("address", agency.address(lang))
      .put("phone", phones)

    val metros = agency.metroIds.flatMap(metroHolder.byCompositeId)
    metros.find(m ⇒ metroColor.isEmpty || m.getColor == metroColor.get).foreach {
      metro ⇒ ans.put("metro", MetroUtil.compositeId(metro))
    }

    optState.foreach { state ⇒
      ans
        .put("is_opened", state.isOpen)
        .put("is_24_hour", state.is24Hour)
        .put("next_event", state.nextEvent.toString)
    }
    agency.url.foreach(url ⇒ ans.put("url", url))
    ans
  }

  def agencyToJson(agencyWithBilling: AgencyWithBilling,
                   metroHolder: MetroHolder,
                   lang: Lang,
                   metroColor: Option[String] = None): JSONObject = {
    val ans = agencyToJson(agencyWithBilling.agency, metroHolder, lang, metroColor)
    val billing = new JSONObject()
    for {
      dumped <- BillingOffers.dumpOffer(agencyWithBilling.billing).toOption
      (key, value) <- dumped
    } billing.put(key, value)
    ans
      .put("billing", billing)
      .put("paid", agencyWithBilling.isPaid)
  }

  def toJson(agencyInRegion: AgenciesInRegion, metroHolder: MetroHolder, lang: Lang, metroColor: Option[String] = None) = {
    val ans = new JSONObject()
      .put("agencies", agenciesToJson(agencyInRegion.sample, metroHolder, lang, metroColor))
      .put("recommended", agenciesToJson(agencyInRegion.recommended, metroHolder, lang, metroColor))
      .put("count", agencyInRegion.count)
      .put("total_count", agencyInRegion.totalCount)
    agencyInRegion.region.foreach { r =>
      ans
        .put("region", regionToJson(r, lang))
        .put("map", toMapJson(r))
    }
    ans
  }

  def toJson(mapInfo: MapRectangle): JSONObject = {
    new JSONObject()
      .put("longitude", mapInfo.lonCenter)
      .put("latitude", mapInfo.latCenter)
      .put("lon_span", mapInfo.lonSpan)
      .put("lat_span", mapInfo.latSpan)
  }

  def toJson(metro: Metro, lang: Lang): JSONObject = {
    val name = LocalizedString.fromLangToVal(metro.getNameList)
    new JSONObject()
      .put("id", MetroUtil.compositeId(metro))
      .put("color", metro.getColor)
      .put("longitude", metro.getPoint.getLongitude)
      .put("latitude", metro.getPoint.getLatitude)
      .put("name", name(lang))
  }

  def toMapJson(region: Region): JSONObject = {
    val bb = region.boundingBox
    new JSONObject()
      .put("longitude", region.longitude)
      .put("latitude", region.latitude)
      .put("lon_span", bb.lonSpan)
      .put("lat_span", bb.latSpan)
  }
}

object CommonJsonSerialization extends CommonJsonSerialization