package ru.yandex.tours.services

import com.google.protobuf.Parser
import com.typesafe.config.Config
import ru.yandex.tours.model.search.{FlightSearchRequest, GetOfferRequest, HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.model.search.SearchResults.{ActualizedOffer, FlightSearchResult, HotelSearchResult, OfferSearchResult}
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray._
import spray.httpx.marshalling._
import spray.http.{HttpRequest, HttpResponse, Uri}
import spray.httpx.unmarshalling.{Deserialized, FromRequestUnmarshaller}
import spray.routing._

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.03.16
 */
class RemoteCalendarPushService(client: AsyncHttpClient, host: String, port: Int, defaultTimeout: FiniteDuration)
                               (implicit ec: ExecutionContext)
  extends RemoteService(client, defaultTimeout) with CalendarPushService {

  private val baseUri = Uri(s"http://$host:$port/$API_VERSION/calendar/push")

  override def saveTourSnippets(request: HotelSearchRequest, response: HotelSearchResult): Future[Unit] = {
    val query = Uri.Query(request.asMap.mapValues(_.toString))
    val uri = baseUri / "tour" / "snippets" withQuery query
    doPostRequest(uri, response.toByteArray)
  }

  override def saveTourOffers(request: OfferSearchRequest, response: OfferSearchResult): Future[Unit] = {
    val query = Uri.Query(request.asMap.mapValues(_.toString))
    val uri = baseUri / "tour" / "offers" withQuery query
    doPostRequest(uri, response.toByteArray)
  }

  override def saveActualization(request: GetOfferRequest, actualized: ActualizedOffer): Future[Unit] = {
    val query = Uri.Query(request.asMap.mapValues(_.toString))
    val uri = baseUri / "tour" / "actualized" withQuery query
    doPostRequest(uri, actualized.toByteArray)
  }

  override def saveRoomSnippets(request: HotelSearchRequest, response: HotelSearchResult): Future[Unit] = {
    val query = Uri.Query(request.asMap.mapValues(_.toString))
    val uri = baseUri / "room" / "snippets" withQuery query
    doPostRequest(uri, response.toByteArray)
  }

  override def saveRoomOffers(request: OfferSearchRequest, response: OfferSearchResult): Future[Unit] = {
    val query = Uri.Query(request.asMap.mapValues(_.toString))
    val uri = baseUri / "room" / "offers" withQuery query
    doPostRequest(uri, response.toByteArray)
  }

  override def saveFlights(request: FlightSearchRequest, response: FlightSearchResult): Future[Unit] = {
    val query = Uri.Query(request.asMap.mapValues(_.toString))
    val uri = baseUri / "flight" / "trips" withQuery query
    doPostRequest(uri, response.toByteArray)
  }
}

object RemoteCalendarPushService {
  import spray.routing.Directives._

  private val hotelId = parameter('hotel_id.as[Int])
  private val tourId = parameter('tour_id)

  private def from[T](parser: Parser[T]) = new FromRequestUnmarshaller[T] {
    override def apply(req: HttpRequest): Deserialized[T] = Right(parser.parseFrom(req.entity.data.toByteArray))
  }

  def routes(calendarPushService: CalendarPushService, routeesContext: RouteesContext)
            (implicit ec: ExecutionContext): Route = {

    implicit val unitMarshaller = ToResponseMarshaller[Future[Unit]] { (res, ctx) =>
      res.onComplete {
        case Success(_) => ctx.marshalTo(HttpResponse.apply())
        case Failure(t) => ctx.handleError(t)
      }
    }

    post {
      path("tour" / "snippets") {
        (routeesContext.searchRequest & entity(from(HotelSearchResult.PARSER))) { (req, res) =>
          complete(calendarPushService.saveTourSnippets(req, res))
        }
      } ~ path("tour" / "offers") {
        (routeesContext.searchRequest & hotelId & entity(from(OfferSearchResult.PARSER))) { (req, hotelId, res) =>
          val request = OfferSearchRequest(req, hotelId)
          complete(calendarPushService.saveTourOffers(request, res))
        }
      } ~ path("tour" / "actualized") {
        (routeesContext.searchRequest & hotelId & tourId & entity(from(ActualizedOffer.PARSER))) {
          (req, hotelId, tourId, res) =>
            val request = GetOfferRequest(OfferSearchRequest(req, hotelId), tourId)
            complete(calendarPushService.saveActualization(request, res))
        }
      } ~ path("room" / "snippets") {
        (routeesContext.searchRequest & entity(from(HotelSearchResult.PARSER))) { (req, res) =>
          complete(calendarPushService.saveRoomSnippets(req, res))
        }
      } ~ path("room" / "offers") {
        (routeesContext.searchRequest & hotelId & entity(from(OfferSearchResult.PARSER))) { (req, hotelId, res) =>
          val request = OfferSearchRequest(req, hotelId)
          complete(calendarPushService.saveRoomOffers(request, res))
        }
      } ~ path("flight" / "trips") {
        (routeesContext.flightSearchRequest & entity(from(FlightSearchResult.PARSER))) { (req, res) =>
          complete(calendarPushService.saveFlights(req, res))
        }
      }
    }
  }

  def fromConfig(client: AsyncHttpClient, config: Config)
                (implicit ec: ExecutionContext): RemoteCalendarPushService = {
    new RemoteCalendarPushService(
      client,
      config.getString("calendar.host"),
      config.getInt("calendar.port"),
      config.getFiniteDuration("calendar.timeout")
    )
  }
}