package ru.yandex.tours.services

import com.typesafe.config.Config
import org.joda.time.LocalDate
import ru.yandex.tours.model.Calendar.{Document, FlightDay, FlightMatrix}
import ru.yandex.tours.model.util.DateInterval
import ru.yandex.tours.util.ProtoIO
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray._
import spray.http.Uri
import spray.routing.Route

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.03.16
 */
class RemoteCalendarService(client: AsyncHttpClient, host: String, port: Int, defaultTimeout: FiniteDuration)
                           (implicit ec: ExecutionContext)
  extends RemoteService(client, defaultTimeout) with CalendarService  {

  private val baseUri = Uri(s"http://$host:$port/$API_VERSION/calendar")

  override def getFlightMatrix: Future[FlightMatrix] = {
    doRequest(baseUri / "matrix", FlightMatrix.parseFrom)
  }

  override def getNearestFlightDay(from: Int, to: Int, when: LocalDate,
                                   interval: DateInterval): Future[Option[FlightDay]] = {
    val uri = (baseUri / "near_flight_day").withQuery(
      "from" -> from.toString,
      "to" -> to.toString,
      "when" -> when.toString,
      "interval_start" -> interval.start.toString,
      "interval_end" -> interval.end.toString
    )
    doRequest(uri, b => if (b.nonEmpty) Some(FlightDay.parseFrom(b)) else None)
  }

  override def getHasFlights(from: Int, to: Int, nights: Option[Int]): Future[Seq[Document]] = {
    val q = Uri.Query.newBuilder
    q += "from" -> from.toString
    q += "to" -> to.toString
    nights.foreach(n => q += "nights" -> n.toString)
    val uri = (baseUri / "has_flights").withQuery(q.result())
    doRequest(uri, ProtoIO.loadFromBytes(_, Document.PARSER).toSeq)
  }

  override def getNoFlights(from: Int, to: Int, nights: Option[Int]): Future[Seq[Document]] = {
    val q = Uri.Query.newBuilder
    q += "from" -> from.toString
    q += "to" -> to.toString
    nights.foreach(n => q += "nights" -> n.toString)
    val uri = (baseUri / "no_flights").withQuery(q.result())
    doRequest(uri, ProtoIO.loadFromBytes(_, Document.PARSER).toSeq)
  }
}

object RemoteCalendarService {
  import ru.yandex.tours.util.spray.SearchDirectives._
  import spray.routing.Directives._

  private val fromAndTo = parameters("from".as[Int], "to".as[Int])

  def routes(calendarService: CalendarService)(implicit ec: ExecutionContext): Route = {
    pathPrefix("matrix") {
      onSuccess(calendarService.getFlightMatrix) {
        matrix => completeProto(matrix)
      }
    } ~ (pathPrefix("near_flight_day") & fromAndTo & date("when") & dateInterval("interval")) {
      (from, to, when, interval) =>
        onSuccess(calendarService.getNearestFlightDay(from, to, when, interval)) {
          fd => completeProto(fd)
        }
    } ~ (pathPrefix("has_flights") & fromAndTo & parameter("nights".as[Int].?)) { (from, to, nights) =>
      onSuccess(calendarService.getHasFlights(from, to, nights)) {
        docs => completeProtoSeq(docs)
      }
    } ~ (pathPrefix("no_flights") & fromAndTo & parameter("nights".as[Int].?)) { (from, to, nights) =>
      onSuccess(calendarService.getNoFlights(from, to, nights)) {
        docs => completeProtoSeq(docs)
      }
    }
  }

  def fromConfig(client: AsyncHttpClient, config: Config)
                (implicit ec: ExecutionContext): RemoteCalendarService = {
    new RemoteCalendarService(
      client,
      config.getString("calendar.host"),
      config.getInt("calendar.port"),
      config.getFiniteDuration("calendar.timeout")
    )
  }
}