package ru.yandex.tours.services

import com.typesafe.config.Config
import ru.yandex.tours.model.Prices.FlightMinPriceMatrix
import ru.yandex.tours.model.search.FlightSearchRequest
import ru.yandex.tours.model.search.SearchResults.FlightSearchResult
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray._
import spray.http.Uri
import spray.routing.{Directives, Route}

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 09.02.16
 */
class RemoteFlightSearchService(client: AsyncHttpClient, host: String, port: Int, defaultTimeout: FiniteDuration)
                               (implicit ec: ExecutionContext)
  extends RemoteService(client, defaultTimeout) with FlightSearchService {

  private val baseUri = Uri(s"http://$host:$port/$API_VERSION/flights")
  private val matrixUri = baseUri / "matrix"

  override def searchFlights(request: FlightSearchRequest, canStartRequest: Boolean): Future[FlightSearchResult] = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(request.hotelRequest)
    query += "can_start_request" -> canStartRequest.toString
    query += "airport_id" -> request.airportId

    doRequest(baseUri.withQuery(query.result()), FlightSearchResult.parseFrom)
  }

  override def getFlightMatrix(request: FlightSearchRequest,
                               from: Seq[Int], airports: Seq[String]): Future[FlightMinPriceMatrix] = {

    val query = Uri.Query.newBuilder
    query ++= toQuery(request.hotelRequest)
    query += "airport_id" -> request.airportId
    query += "airports" -> airports.mkString(",")
    query += "departures" -> from.mkString(",")

    doRequest(matrixUri.withQuery(query.result()), FlightMinPriceMatrix.parseFrom)
  }
}

object RemoteFlightSearchService extends Directives with SearchDirectives {

  private val canStartRequest = parameter('can_start_request.as[Boolean])

  def routes(flightSearchService: FlightSearchService, routeesContext: RouteesContext)
            (implicit ec: ExecutionContext): Route = {
    (pathPrefix("flights") & routeesContext.flightSearchRequest) { flightRequest =>
      (pathEndOrSingleSlash & canStartRequest) { canStartRequest =>
        onSuccess(flightSearchService.searchFlights(flightRequest, canStartRequest)) {
          result => completeProto(result)
        }
      } ~ (path("matrix") & intArray("departures") & array("airports")) { (from, airports) =>
        onSuccess(flightSearchService.getFlightMatrix(flightRequest, from, airports)) {
          result => completeProto(result)
        }
      }
    }
  }

  def fromConfig(client: AsyncHttpClient, config: Config)
                (implicit ec: ExecutionContext): RemoteFlightSearchService = {
    new RemoteFlightSearchService(
      client,
      config.getString("searcher.host"),
      config.getInt("searcher.port"),
      config.getFiniteDuration("searcher.timeout")
    )
  }
}