package ru.yandex.tours.services

import com.typesafe.config.Config
import ru.yandex.tours.model.search.SearchResults.{HotelSearchResult, OfferSearchResult}
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.util.ProtoIO
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import spray.http.Uri

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

class RemoteHotelSearchService(client: AsyncHttpClient, host: String, port: Int,
                               timeout: FiniteDuration, prefix: String)
                              (implicit ec: ExecutionContext)
  extends RemoteService(client, timeout) with HotelSearchService {

  private def searchRequest(request: HotelSearchRequest, canStartRequest: Boolean, onlyFromLongCache: Boolean) = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(request)
    query += "can_start_request" -> canStartRequest.toString
    query += "only_from_long_cache" -> onlyFromLongCache.toString
    Uri(s"http://$host:$port/$API_VERSION/$prefix/search")
      .withQuery(query.result())
  }

  private def searchRequest(request: HotelSearchRequest, hotelIds: Iterable[Int], canStartRequest: Boolean) = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(request)
    query += "can_start_request" -> canStartRequest.toString
    query += "hotel_ids" -> hotelIds.mkString(",")
    Uri(s"http://$host:$port/$API_VERSION/$prefix/search/hotels")
      .withQuery(query.result())
  }

  private def searchRequest(request: OfferSearchRequest, canStartRequest: Boolean) = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(request.hotelRequest)
    query += "can_start_request" -> canStartRequest.toString
    Uri(s"http://$host:$port/$API_VERSION/$prefix/hotel/${request.hotelId}")
      .withQuery(query.result())
  }

  override def search(request: HotelSearchRequest, canStartRequest: Boolean,
                      onlyFromCache: Boolean): Future[HotelSearchResult] = {
    doRequest(searchRequest(request, canStartRequest, onlyFromCache), HotelSearchResult.parseFrom)
  }

  override def searchHotel(request: OfferSearchRequest, canStartRequest: Boolean): Future[OfferSearchResult] = {
    doRequest(searchRequest(request, canStartRequest), OfferSearchResult.parseFrom)
  }

  override def searchHotels(request: HotelSearchRequest, hotelIds: Iterable[Int],
                            canStartRequest: Boolean): Future[Seq[OfferSearchResult]] = {
    def parser(bytes: Array[Byte]) = ProtoIO.loadFromBytes(bytes, OfferSearchResult.PARSER).toSeq
    doRequest(searchRequest(request, hotelIds, canStartRequest), parser)
  }
}

object RemoteHotelSearchService {
  def fromConfig(client: AsyncHttpClient, config: Config, prefix: String = "")
                (implicit ec: ExecutionContext): RemoteHotelSearchService = {
    new RemoteHotelSearchService(
      client,
      config.getString("searcher.host"),
      config.getInt("searcher.port"),
      config.getFiniteDuration("searcher.timeout"),
      prefix
    )
  }
}