package ru.yandex.tours.services

import com.typesafe.config.Config
import org.joda.time.LocalDate
import ru.yandex.tours.model.Prices.{DirectionBestPrice, PriceGraph}
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray._
import spray.http.Uri
import spray.routing.{Directive1, Directives, Route}

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

class RemoteMinPriceService(client: AsyncHttpClient, host: String, port: Int, timeout: FiniteDuration, prefix: String)
                           (implicit ec: ExecutionContext) extends RemoteService(client, timeout) with MinPriceService {

  val baseUrl = s"http://$host:$port/$API_VERSION/$prefix"

  val minPriceUri = Uri(s"$baseUrl/min_prices")
  val minPriceIntervalUri = Uri(s"$baseUrl/min_prices_interval")

  val directionBestPriceUri = Uri(s"$baseUrl/best_direction_price")

  private def minimalPrice(request: HotelSearchRequest, size: Int): Uri =
    minPriceUri.withQuery(("size" -> size.toString) +: toQuery(request))

  private def minimalPriceInterval(from: Int, to: Int, when: LocalDate, nights: Int, ages: Seq[Int],
                                   graphStart: LocalDate, graphEnd: LocalDate): Uri = {
    val queryBuilder = Uri.Query.newBuilder
    queryBuilder ++= Seq(
      "from" -> from.toString,
      "to" -> to.toString,
      "when" -> formatter.print(when),
      "nights" -> nights.toString,
      "graphStart" -> formatter.print(graphStart),
      "graphEnd" -> formatter.print(graphEnd)
    )
    for (age <- ages) {
      queryBuilder += "ages" -> age.toString
    }
    minPriceIntervalUri.withQuery(queryBuilder.result())
  }

  override def getMinPriceGraph(request: HotelSearchRequest, graphLength: Int): Future[PriceGraph] =
    doRequest(minimalPrice(request, graphLength), PriceGraph.parseFrom)

  override def getMinPriceGraph(from: Int, to: Int, when: LocalDate, nights: Int, ages: Seq[Int],
                                graphStart: LocalDate, graphEnd: LocalDate): Future[PriceGraph] = {
    doRequest(minimalPriceInterval(from, to, when, nights, ages, graphStart, graphEnd), PriceGraph.parseFrom)
  }

  override def getDirectionMinPrice(from: Int, to: Int): Future[Option[DirectionBestPrice]] = {
    val request = directionBestPriceUri.withQuery(
      "from" -> from.toString,
      "to" -> to.toString
    )
    doRequest(request, bytes => if (bytes.isEmpty) None else Some(DirectionBestPrice.parseFrom(bytes)))
  }
}

object RemoteMinPriceService {

  import SearchDirectives._
  import Directives._

  private val graphSize = parameter('size.as[Int] ? MinPriceService.DEFAULT_GRAPH_LENGTH)
  private val bestDirectionPriceParameters = parameters('from.as[Int], 'to.as[Int])

  def fromConfig(client: AsyncHttpClient, config: Config, prefix: String)
                (implicit ec: ExecutionContext): RemoteMinPriceService = {
    new RemoteMinPriceService(
      client,
      config.getString("host"),
      config.getInt("port"),
      config.getFiniteDuration("timeout"),
      prefix
    )
  }

  def routes(minPriceService: MinPriceService, requestDirective: Directive1[HotelSearchRequest])
            (implicit ec: ExecutionContext): Route = {
    (path("min_prices") & requestDirective & graphSize) { (request, size) =>
      onSuccess(minPriceService.getMinPriceGraph(request, size)) {
        graph => completeProto(graph)
      }
    } ~ (path("min_prices_interval") & requestDirective & date("graphStart") & date("graphEnd")) {
      (request, graphStart, graphEnd) =>
        val future = minPriceService.getMinPriceGraph(request.from, request.to, request.when, request.nights,
          request.ages.toSeq, graphStart, graphEnd)
        onSuccess(future) {
          graph => completeProto(graph)
        }
    } ~ (path("best_direction_price") & bestDirectionPriceParameters) { (from, to) =>
      onSuccess(minPriceService.getDirectionMinPrice(from, to)) { bestPrice =>
        completeProto(bestPrice)
      }
    }
  }
}