package ru.yandex.tours.services

import com.typesafe.config.Config
import ru.yandex.tours.model.Prices.Recommendation
import ru.yandex.tours.model.search.WhereToGoRequest
import ru.yandex.tours.model.util.SortType.SortType
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray._
import spray.http.Uri
import spray.routing.{Directives, Route}

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

class RemoteRecommendService(client: AsyncHttpClient, host: String, port: Int, timeout: FiniteDuration, prefix: String)
                            (implicit ec: ExecutionContext)
  extends RemoteService(client, timeout) with RecommendService {

  private val baseUri = Uri(s"http://$host:$port/$API_VERSION/$prefix/recommend")
  private val directionsUri = Uri(s"http://$host:$port/$API_VERSION/$prefix/recommend/directions")
  private val allDirectionsUri = Uri(s"http://$host:$port/$API_VERSION/$prefix/recommend/directions/all")
  private val resortsUri = Uri(s"http://$host:$port/$API_VERSION/$prefix/recommend/resorts")

  private def toQuery(userIdentifiers: UserIdentifiers): Uri.Query = {
    val params = userIdentifiers.login.map("login" -> _) ++
      userIdentifiers.uid.map("uid" -> _) ++
      userIdentifiers.yuid.map("yuid" -> _) ++
      userIdentifiers.yuid.map("user_id" -> _)
    Uri.Query(params.toSeq: _*)
  }

  private def toQuery(request: WhereToGoRequest): Uri.Query = {
    val params = Map(
      "thematic" -> request.thematic.toString,
      "budget" -> request.budget.fold("")(_.toString),
      "month" -> request.month.fold("")(_.getValue.toString),
      "no_visa" -> request.noVisa.toString,
      "country" -> request.countryId.fold("")(_.toString)
    ).filter(_._2.nonEmpty)
    Uri.Query(params.toSeq: _*)
  }

  private def createRequest(userIdentifiers: UserIdentifiers, geoId: Int): Uri = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(userIdentifiers)
    query += "geo_id" -> geoId.toString
    baseUri.withQuery(query.result())
  }

  private def directions(userIdentifiers: UserIdentifiers, geoId: Int, sortBy: SortType): Uri = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(userIdentifiers)
    query += "geo_id" -> geoId.toString
    query += "sort" -> sortBy.toString
    directionsUri.withQuery(query.result())
  }
  private def allDirections(userIdentifiers: UserIdentifiers,
                            geoId: Int, sortBy: SortType,
                            branding: Option[String]): Uri = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(userIdentifiers)
    query += "geo_id" -> geoId.toString
    query += "sort" -> sortBy.toString
    branding.foreach(query += "branding" -> _)
    allDirectionsUri.withQuery(query.result())
  }
  private def resorts(countryId: Int,
                      userIdentifiers: UserIdentifiers,
                      geoId: Int): Uri = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(userIdentifiers)
    query += "geo_id" -> geoId.toString
    query += "country_id" -> countryId.toString
    resortsUri.withQuery(query.result())
  }

  private def directions(request: WhereToGoRequest, userIdentifiers: UserIdentifiers,
                         geoId: Int, sortBy: SortType): Uri = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(request)
    query ++= toQuery(userIdentifiers)
    query += "geo_id" -> geoId.toString
    query += "sort" -> sortBy.toString
    directionsUri.withQuery(query.result())
  }

  override def recommendTopDirections(userIdentifiers: UserIdentifiers, userRegion: Int): Future[Recommendation] =
    doRequest(createRequest(userIdentifiers, userRegion), Recommendation.parseFrom)

  override def recommendDirections(userIdentifiers: UserIdentifiers,
                                   userRegion: Int, sortBy: SortType): Future[Recommendation] = {
    doRequest(directions(userIdentifiers, userRegion, sortBy), Recommendation.parseFrom)
  }

  override def recommendAllDirections(userIdentifiers: UserIdentifiers,
                                      userRegion: Int, sortBy: SortType,
                                      branding: Option[String]): Future[Recommendation] = {
    doRequest(allDirections(userIdentifiers, userRegion, sortBy, branding), Recommendation.parseFrom)
  }

  override def recommendResorts(countryId: Int,
                                userIdentifiers: UserIdentifiers,
                                userRegion: Int): Future[Recommendation] = {
    doRequest(resorts(countryId, userIdentifiers, userRegion), Recommendation.parseFrom)
  }

  override def recommendDirections(request: WhereToGoRequest, userIdentifiers: UserIdentifiers,
                                   userRegion: Int, sortBy: SortType): Future[Recommendation] = {
    doRequest(directions(request, userIdentifiers, userRegion, sortBy), Recommendation.parseFrom)
  }
}

object RemoteRecommendService extends Directives with SearchDirectives with CommonModelDirectives {
  def routes(recommendService: RecommendService)(implicit ec: ExecutionContext): Route = {
    (path("recommend") & geoId & userIdentifiers) { (geoId, userIdentifiers) =>
      onSuccess(recommendService.recommendTopDirections(userIdentifiers, geoId)) {
        recommendation => completeProto(recommendation)
      }
    } ~ (path("recommend" / "directions") & whereToGoRequest & userIdentifiers & geoId & sortBy) { (request, userIdentifiers, geoId, sortBy) =>
      onSuccess(recommendService.recommendDirections(request, userIdentifiers, geoId, sortBy)) {
        recommendation => completeProto(recommendation)
      }
    } ~ (path("recommend" / "directions") & userIdentifiers & geoId & sortBy) { (userIdentifiers, geoId, sortBy) =>
      onSuccess(recommendService.recommendDirections(userIdentifiers, geoId, sortBy)) {
        recommendation => completeProto(recommendation)
      }
    } ~ (path("recommend" / "resorts") & parameter("country_id".as[Int]) & userIdentifiers & geoId) {
      (countryId, userIdentifiers, geoId) =>
        onSuccess(recommendService.recommendResorts(countryId, userIdentifiers, geoId)) {
          recommendation => completeProto(recommendation)
        }
    } ~ (path("recommend" / "directions" / "all") & userIdentifiers & geoId & sortBy & branding) {
      (userIdentifiers, geoId, sortBy, branding) =>
      onSuccess(recommendService.recommendAllDirections(userIdentifiers, geoId, sortBy, branding)) {
        recommendation => completeProto(recommendation)
      }
    }
  }

  def fromConfig(client: AsyncHttpClient, config: Config, prefix: String)
                (implicit ec: ExecutionContext): RemoteRecommendService = {
    new RemoteRecommendService(
      client,
      config.getString("searcher.host"),
      config.getInt("searcher.port"),
      config.getFiniteDuration("searcher.timeout"),
      prefix
    )
  }
}