package ru.yandex.tours.services

import akka.util.Timeout
import org.joda.time.format.DateTimeFormat
import ru.yandex.tours.model.search.{EmptySearchFilter, HotelSearchRequest, SearchFilter}
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.Uri

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

class RemoteService(client: AsyncHttpClient, defaultTimeout: FiniteDuration)
                   (implicit ec: ExecutionContext) {

  protected val formatter = DateTimeFormat.forPattern("yyyy-MM-dd")
  protected val API_VERSION = "api/1.x"

  final protected def toQuery(request: HotelSearchRequest): Uri.Query = {
    val queryBuilder = Uri.Query.newBuilder
    queryBuilder ++= Seq(
      "from" -> request.from.toString,
      "to" -> request.to.toString,
      "nights" -> request.nights.toString,
      "when" -> formatter.print(request.when),
      "when_flex" -> request.flexWhen.toString,
      "nights_flex" -> request.flexNights.toString,
      "currency" → request.currency.name(),
      "lang" → request.lang.toString,
      "unique_id" → request.uniqueId
    ) ++ request.utmAsMap.filter(kv => kv._2 != "")
    for (age <- request.ages) {
      queryBuilder += "ages" -> age.toString
    }
    if (request.filter != EmptySearchFilter) {
      queryBuilder += SearchFilter.name -> request.filter.toString
    }
    queryBuilder.result()
  }

  final protected def doRequest[T](uri: Uri, parser: Array[Byte] => T,
                                   timeout: FiniteDuration = defaultTimeout): Future[T] = {
    for ((status, bytes) <- client.getBytes(uri)(new Timeout(timeout))) yield {
      if (status.isSuccess) {
        parser(bytes)
      } else {
        throw new RuntimeException(s"Failed to do remote request. Status = $status. Request: $uri")
      }
    }
  }

  final protected def doPostRequest[T](uri: Uri, bytes: Array[Byte],
                                       timeout: FiniteDuration = defaultTimeout): Future[Unit] = {

    for ((status, _) <- client.post(uri, bytes)(new Timeout(timeout))) yield {
      if (status.isSuccess) {
        ()
      } else {
        throw new RuntimeException(s"Failed to do remote POST request. Status = $status. Request: $uri")
      }
    }
  }
}
