package ru.yandex.tours.services

import com.typesafe.config.Config
import ru.yandex.tours.model.search.GetOfferRequest
import ru.yandex.tours.model.search.SearchResults.ActualizedOffer
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import spray.http.Uri

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 20.01.15
 */
class RemoteToursSearchService(client: AsyncHttpClient, host: String, port: Int,
                               timeout: FiniteDuration, prefix: String)
                              (implicit ec: ExecutionContext)
  extends RemoteHotelSearchService(client, host, port, timeout, prefix) with OfferSearchService {

  private def actualizeRequest(request: GetOfferRequest) = {
    val hotelId = request.toursInHotelRequest.hotelId
    val tourId = request.tourId
    Uri(s"http://$host:$port/$API_VERSION/$prefix/hotel/$hotelId/tour/$tourId/actualize")
      .withQuery(toQuery(request.hotelRequest))
  }

  private def searchRequest(request: GetOfferRequest) = {
    val hotelId = request.toursInHotelRequest.hotelId
    val tourId = request.tourId
    Uri(s"http://$host:$port/$API_VERSION/$prefix/hotel/$hotelId/tour/$tourId")
      .withQuery(toQuery(request.hotelRequest))
  }

  override def searchOffer(request: GetOfferRequest): Future[Option[ActualizedOffer]] = {
    doRequest(searchRequest(request), bytes => if (bytes.isEmpty) None else Some(ActualizedOffer.parseFrom(bytes)))
  }

  override def actualizeOffer(request: GetOfferRequest): Future[Option[ActualizedOffer]] = {
    doRequest(actualizeRequest(request), bytes => if (bytes.isEmpty) None else Some(ActualizedOffer.parseFrom(bytes)),
      30.seconds)
  }
}

object RemoteToursSearchService {
  def fromConfig(client: AsyncHttpClient, config: Config, prefix: String)
                (implicit ec: ExecutionContext): RemoteToursSearchService = {
    new RemoteToursSearchService(
      client,
      config.getString("searcher.host"),
      config.getInt("searcher.port"),
      config.getFiniteDuration("searcher.timeout"),
      prefix
    )
  }
}
