package ru.yandex.tours.services

import com.typesafe.config.Config
import ru.yandex.tours.model.search.SearchResults.TransferSearchResult
import ru.yandex.tours.model.search.TransferSearchRequest
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray._
import spray.http.Uri
import spray.routing._

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration.FiniteDuration

/**
  * Created by asoboll on 25.04.16.
  */
class RemoteTransferSearchService(client: AsyncHttpClient, host: String, port: Int, defaultTimeout: FiniteDuration)
                                 (implicit ec: ExecutionContext)
  extends RemoteService(client, defaultTimeout) with TransferSearchService {

  private val baseUri = Uri(s"http://$host:$port/$API_VERSION/${RemoteTransferSearchService.routePrefix}")

  override def searchTransfers(request: TransferSearchRequest,
                               canStartRequest: Boolean): Future[TransferSearchResult] = {
    val query = Uri.Query.newBuilder
    query ++= toQuery(request.hotelRequest)
    query += "can_start_request" -> canStartRequest.toString
    query += "airport_id" -> request.airportId
    query += "hotel_id" -> request.hotelId.toString
    request.userIp.foreach(query += "user_ip" -> _)

    doRequest(baseUri.withQuery(query.result()), TransferSearchResult.parseFrom)
  }
}

object RemoteTransferSearchService extends Directives with SearchDirectives {
  val routePrefix = "transfers"

  private val canStartRequest = parameter('can_start_request.as[Boolean] ? true)

  def routes(transferSearchService: TransferSearchService, routeesContext: RouteesContext)
            (implicit ec: ExecutionContext): Route = {
    (pathPrefix(routePrefix) & routeesContext.transferSearchRequest & canStartRequest) { (transferRequest, canStartRequest) =>
      onSuccess(transferSearchService.searchTransfers(transferRequest, canStartRequest)) {
        result => completeProto(result)
      }
    }
  }

  def fromConfig(client: AsyncHttpClient, config: Config)
                (implicit ec: ExecutionContext): RemoteTransferSearchService = {
    new RemoteTransferSearchService(
      client,
      config.getString("searcher.host"),
      config.getInt("searcher.port"),
      config.getFiniteDuration("searcher.timeout")
    )
  }
}