package ru.yandex.tours.services

import akka.util.Timeout
import com.google.protobuf.Parser
import com.typesafe.config.Config
import ru.yandex.tours.model.UserVisits.{PriceData, Record}
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.util.ProtoIO
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray._
import spray.http.{HttpRequest, StatusCodes, Uri}
import spray.httpx.unmarshalling._
import spray.routing.{Directives, Route}

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration.FiniteDuration

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.06.16
 */
class RemoteUserFavoritesService(client: AsyncHttpClient, host: String, port: Int,
                                 timeout: FiniteDuration)
                                (implicit ec: ExecutionContext)
  extends RemoteService(client, timeout) with UserFavoritesService {

  private implicit val implicitTimeout = new Timeout(timeout)

  private val baseUri = Uri(s"http://$host:$port/$API_VERSION/user/favorites")

  override def getFavorites(uid: Long, hotelIds: Seq[Int]): Future[Seq[Record]] = {
    val req = baseUri / uid withQuery ("hotel_ids" → hotelIds.mkString(","))
    doRequest(req, ProtoIO.loadFromBytes(_, Record.PARSER).toVector)
  }

  override def getFavorites(uid: Long): Future[Seq[Record]] = {
    val req = baseUri / uid
    doRequest(req, ProtoIO.loadFromBytes(_, Record.PARSER).toVector)
  }

  override def deleteFavorites(uid: Long, hotelId: Int): Future[Unit] = {
    val req = baseUri / uid / hotelId
    for ((status, body) ← client.delete(req)) yield {
      require(status.isSuccess, s"Failed to do DELETE request. Status = $status. request: $req, response = $body")
      ()
    }
  }

  override def putVisit(uid: Long, hotelId: Int, request: HotelSearchRequest, data: PriceData): Future[Unit] = {
    val req = baseUri / uid / hotelId withQuery toQuery(request)
    for ((status, recordId) ← client.put(req, data.toByteArray)) yield {
      require(status.isSuccess, s"Failed to do PUT request. Status = $status. Request: $req, response = $recordId")
      ()
    }
  }
}

object RemoteUserFavoritesService {

  import SearchDirectives._
  import Directives._

  private def from[T](parser: Parser[T]) = new FromRequestUnmarshaller[T] {
    override def apply(req: HttpRequest): Deserialized[T] = Right(parser.parseFrom(req.entity.data.toByteArray))
  }

  def fromConfig(client: AsyncHttpClient, config: Config)
                (implicit ec: ExecutionContext): RemoteUserFavoritesService = {
    new RemoteUserFavoritesService(
      client,
      config.getString("host"),
      config.getInt("port"),
      config.getFiniteDuration("timeout")
    )
  }

  def routes(userFavoritesService: UserFavoritesService, routeesContext: RouteesContext)
            (implicit ec: ExecutionContext): Route = {
    pathPrefix(LongNumber) { uid ⇒
      (get & pathEndOrSingleSlash & intArray("hotel_ids", isEmptyOk = true)) { hotelIds ⇒
        val future =
          if (hotelIds.isEmpty) userFavoritesService.getFavorites(uid)
          else userFavoritesService.getFavorites(uid, hotelIds)
        onSuccess(future) { records ⇒ completeProtoSeq(records) }
      } ~ path(IntNumber) { hotelId ⇒
        (put & routeesContext.searchRequest & entity(from(PriceData.PARSER))) { (request, data) ⇒
          onSuccess(userFavoritesService.putVisit(uid, hotelId, request, data)) {
            _ ⇒ complete(StatusCodes.OK)
          }
        } ~ delete {
          onSuccess(userFavoritesService.deleteFavorites(uid, hotelId)) { _ ⇒ complete(StatusCodes.OK) }
        }
      }
    }
  }
}