package ru.yandex.tours.services

import akka.util.Timeout
import com.google.protobuf.Parser
import com.typesafe.config.Config
import ru.yandex.tours.model.UserVisits.{PriceData, Record}
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.util.ProtoIO
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray._
import spray.http.{HttpRequest, StatusCodes, Uri}
import spray.httpx.unmarshalling._
import spray.routing.{Directives, Route}

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.06.16
 */
class RemoteUserHistoryService(client: AsyncHttpClient, host: String, port: Int,
                               timeout: FiniteDuration)
                              (implicit ec: ExecutionContext)
  extends RemoteService(client, timeout) with UserHistoryService {

  private implicit val implicitTimeout = new Timeout(timeout)

  private val baseUri = Uri(s"http://$host:$port/$API_VERSION/user/history")

  override def getRecentItems(uid: Long, limit: Int): Future[Seq[Record]] = {
    val req = baseUri / uid withQuery ("limit" → limit.toString)
    doRequest(req, ProtoIO.loadFromBytes(_, Record.PARSER).toVector)
  }


  override def updateVisit(uid: Long, hotelId: Int, request: HotelSearchRequest, data: PriceData): Future[Unit] = {
    Future.successful(())
  }

  override def putVisit(uid: Long, hotelId: Int, request: HotelSearchRequest, data: PriceData): Future[Unit] = {
    val req = baseUri / uid / hotelId withQuery toQuery(request)
    for ((status, response) ← client.put(req, data.toByteArray)) yield {
      require(status.isSuccess, s"Failed to do PUT request. Status = $status. Request: $req, response = $response")
    }
  }
}

object RemoteUserHistoryService {

  import Directives._

  private def from[T](parser: Parser[T]) = new FromRequestUnmarshaller[T] {
    override def apply(req: HttpRequest): Deserialized[T] = Right(parser.parseFrom(req.entity.data.toByteArray))
  }

  def fromConfig(client: AsyncHttpClient, config: Config)
                (implicit ec: ExecutionContext): RemoteUserHistoryService = {
    new RemoteUserHistoryService(
      client,
      config.getString("host"),
      config.getInt("port"),
      config.getFiniteDuration("timeout")
    )
  }

  def routes(userHistoryService: UserHistoryService, routeesContext: RouteesContext)
            (implicit ec: ExecutionContext): Route = {
    pathPrefix(LongNumber) { uid ⇒
      (get & pathEndOrSingleSlash & parameter("limit".as[Int])) { limit ⇒
        val future = userHistoryService.getRecentItems(uid, limit)
        onSuccess(future) { records ⇒ completeProtoSeq(records) }
      } ~ (put & path(IntNumber)) { hotelId ⇒
        (routeesContext.searchRequest & entity(from(PriceData.PARSER))) { (request, data) ⇒
          onSuccess(userHistoryService.putVisit(uid, hotelId, request, data)) { _ ⇒
            complete(StatusCodes.OK)
          }
        }
      }
    }
  }
}