package ru.yandex.tours.services.users

import akka.util.Timeout
import com.google.common.base.Charsets
import com.typesafe.config.Config
import org.apache.commons.codec.binary.Base64
import play.api.libs.json.Json
import ru.yandex.tours.model.UserVisits.{PriceData, Record}
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.search.SearchRequests.ProtoHotelSearchRequest
import ru.yandex.tours.services.UserHistoryService
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.spray.RichUri
import spray.http.{StatusCodes, Uri}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 31.08.16
 */
class UserHistoryServiceImpl(httpClient: AsyncHttpClient,
                             host: String,
                             port: Int,
                             timeout: FiniteDuration)
                            (implicit ec: ExecutionContext) extends UserHistoryService {

  import UserHistoryServiceImpl._

  private val baseUri = Uri(s"http://$host:$port/history/1.0/travel:hotels")
  private implicit val iTimeout = Timeout(timeout)

  private def jsonToRecords(json: String): Seq[Record] = {
    val entities = Json.parse(json).as(responseReads).entities.getOrElse("hotels", Seq.empty)
    for (entity ← entities) yield {
      val payload = Json.parse(entity.payload).as(payloadFormat)

      Record.newBuilder()
        .setRecordId(entity.entity_id)
        .setCreated(entity.visit_timestamp)
        .setHotelId(entity.entity_id.toInt)
        .setRequest(payload.protoRequest)
        .setPrices(payload.protoPriceData)
        .build()
    }
  }

  private def toPayload(request: HotelSearchRequest, data: PriceData): String = {
    Json.toJson(
      Payload(
        Base64.encodeBase64String(request.toProto.toByteArray),
        Base64.encodeBase64String(data.toByteArray)
      )
    ).toString()
  }

  override def getRecentItems(uid: Long, limit: Int): Future[Seq[Record]] = {
    httpClient.get(baseUri / s"uid:$uid").map {
      case (StatusCodes.OK, json) ⇒
        jsonToRecords(json)
      case (status, body) ⇒
        sys.error(s"Unexpected response from personal-api. Status = $status, body = $body")
    }
  }


  override def updateVisit(uid: Long, hotelId: Int, request: HotelSearchRequest, data: PriceData): Future[Unit] = {
    val payload = toPayload(request, data).getBytes(Charsets.UTF_8)
    httpClient.post(baseUri / s"uid:$uid" / hotelId, payload).map {
      case (StatusCodes.OK, json) ⇒
      case (status, body) ⇒
        sys.error(s"Unexpected response from personal-api. Status = $status, body = $body")
    }
  }

  override def putVisit(uid: Long, hotelId: Int, request: HotelSearchRequest, data: PriceData): Future[Unit] = {
    val payload = toPayload(request, data).getBytes(Charsets.UTF_8)
    httpClient.put(baseUri / s"uid:$uid" / hotelId, payload).map {
      case (StatusCodes.OK, json) ⇒
      case (status, body) ⇒
        sys.error(s"Unexpected response from personal-api. Status = $status, body = $body")
    }
  }
}

object UserHistoryServiceImpl {

  def fromConfig(client: AsyncHttpClient, config: Config)
                (implicit ec: ExecutionContext): UserHistoryService = {
    new UserHistoryServiceImpl(
      client,
      config.getString("host"),
      config.getInt("port"),
      config.getFiniteDuration("timeout")
    )
  }

  case class Response(entities: Map[String, Seq[Entity]])

  case class Entity(entity_id: String, visit_timestamp: Long, payload: String)

  case class Payload(request: String, data: String) {
    def protoRequest: ProtoHotelSearchRequest = ProtoHotelSearchRequest.parseFrom(Base64.decodeBase64(request))
    def protoPriceData: PriceData = PriceData.parseFrom(Base64.decodeBase64(data))
  }

  implicit val entityReads = Json.reads[Entity]
  implicit val responseReads = Json.reads[Response]

  implicit val payloadFormat = Json.format[Payload]

}