package ru.yandex.tours.subscriptions.model

import org.joda.time.DateTime
import ru.yandex.tours.filter.Filters
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.subscriptions.Notification.Offer
import ru.yandex.tours.model.subscriptions._
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.spray._
import ru.yandex.tours.util.{Logging, spray => spray_utils}
import shapeless._
import spray.http.Uri
import spray.routing.Directives._

import scala.collection.JavaConverters._
import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 04.09.15
 */
object ModelConverters extends Logging with CommonDirectives with SearchDirectives {

  implicit class RichSubscription(subscription: Subscription) {
    def toProto = subscriptionToProto(subscription)
  }
  implicit class RichProtoSubscription(subscription: Subscriptions.Subscription) {
    def toModel = protoToSubscription(subscription)
  }
  implicit class RichNotification(notification: Notification) {
    def toProto = notificationToProto(notification)
  }
  implicit class RichProtoNotification(notification: Subscriptions.Notification) {
    def toModel(hotelsIndex: HotelsIndex, tourOperators: TourOperators) =
      protoToNotification(notification, hotelsIndex, tourOperators)
  }
  implicit class RichQuery(query: Uri.Query) {
    def toSubject = queryToSubject(query)
  }

  def subscriptionToProto(subscription: Subscription): Subscriptions.Subscription = {
    Subscriptions.Subscription.newBuilder()
      .setUser(subscription.user.toString)
      .setId(subscription.id)
      .setEmail(subscription.email)
      .setIntervalInMinutes(subscription.interval.toMinutes)
      .setState(subscription.state)
      .setEnabled(subscription.enabled)
      .setSubjectQuery(subscription.subject.query)
      .build()
  }

  def protoToSubscription(subscription: Subscriptions.Subscription): Subscription = {
    Subscription(
      UserIdentity.fromString(subscription.getUser),
      subscription.getId,
      subscription.getEmail,
      subscription.getIntervalInMinutes.minutes,
      subscription.getState,
      Uri.Query(subscription.getSubjectQuery).toSubject
    )
  }

  def notificationToProto(notification: Notification): Subscriptions.Notification = {
    val builder = Subscriptions.Notification.newBuilder()
      .setCreated(notification.created.getMillis)
      .setSubscription(notification.subscription.toProto)

    def populate(builder: Subscriptions.Notification.HotelOffers.Builder, hotelOffers: Notification.HotelOffers): Unit = {
      builder.setHotelId(hotelOffers.hotel.id)
      for (offer <- hotelOffers.offers) {
        builder.addOfferBuilder()
          .setWhen(offer.when.toMillis)
          .setNights(offer.nights)
          .setPansion(offer.pansion)
          .setTourOperator(offer.operator.id)
          .setPrice(offer.price)
          .setCreated(offer.created.getMillis)
      }
    }

    notification.payload match {
      case Notification.DirectionOffers(offers) =>
        val b = builder.getDirectionOffersBuilder
        for (hotelOffers <- offers) {
          populate(b.addOfferBuilder(), hotelOffers)
        }
      case hotelOffers: Notification.HotelOffers =>
        populate(builder.getHotelOffersBuilder, hotelOffers)
      case Notification.Unknown =>
    }

    builder.build
  }

  def protoToNotification(notification: Subscriptions.Notification, hotelsIndex: HotelsIndex, tourOperators: TourOperators)
                         : Notification = {
    val subscription = notification.getSubscription.toModel

    def protoToOffer(offer: Subscriptions.Notification.Offer): Option[Offer] = {
      for (operator <- tourOperators.getById(offer.getTourOperator)) yield {
        Notification.Offer(
          offer.getWhen.toLocalDate,
          offer.getNights,
          offer.getPansion,
          operator,
          offer.getPrice,
          new DateTime(offer.getCreated)
        )
      }
    }

    def protoToHotelOffers(hotelOffers: Subscriptions.Notification.HotelOffers): Option[Notification.HotelOffers] = {
      val hotelId = hotelOffers.getHotelId
      hotelsIndex.getHotelById(hotelId) match {
        case Some(hotel) =>
          val offers = hotelOffers.getOfferList.asScala.flatMap(protoToOffer)
          Some(Notification.HotelOffers(hotel, offers))
        case None =>
          log.warn(s"Unknown hotel $hotelId in notification for $subscription")
          None
      }
    }

    val payload =
      if (notification.hasDirectionOffers) {
        val offers = notification.getDirectionOffers.getOfferList.asScala.flatMap(protoToHotelOffers)
        Notification.DirectionOffers(offers)
      } else if (notification.hasHotelOffers) {
        val hotelOffers = notification.getHotelOffers
        protoToHotelOffers(hotelOffers).getOrElse {
          Notification.Unknown
        }
      } else {
        Notification.Unknown
      }

    Notification(
      new DateTime(notification.getCreated),
      subscription,
      payload
    )
  }

  def queryToSubject(query: Uri.Query): Subscription.Subject = {
    val directive = (parameters('from.as[Int]) & intArray("ages") & searchDates & Filters.filters) hflatMap {
      case from :: ages :: dates :: filters :: HNil =>
        provide(Subscription.SubjectParams(from, ages, dates, filters.filter(_.nonEmpty)))
    }

    val params = spray_utils.extract(query, directive)
    spray_utils.hextract(query, parameters('to.as[Int].?, 'hotel_id.as[Int].?)) match {
      case _ :: Some(hotelId) :: HNil => Subscription.Hotel(query.toString(), hotelId, params)
      case Some(to) :: _ :: HNil => Subscription.Direction(query.toString(), to, params)
    }
  }

}
