package ru.yandex.tours.tanker

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDef, DataTypes}
import ru.yandex.tours.model.hotels.HotelsHolder.HotelType
import _root_.ru.yandex.tours.model.hotels.Features._
import ru.yandex.tours.util.Speller
import ru.yandex.tours.model.Languages._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.03.15
 *
 * @param map (keySet, key, lang) -> value map
 */
case class Translations(map: Map[(String, String, Lang), String]) {
  def translate(keySet: String, key: String, lang: Lang): Option[String] = map.get((keySet, key, lang))
  def translate(keySet: String, key: String, lang: Lang, default: String): String =
    map.getOrElse((keySet, key, lang), default)

  private def tr(keySet: String, value: String, lang: Lang): Option[String] =
    translate(keySet, "full_" + value, lang) //hack
      .orElse(translate(keySet, value, lang))

  def translateFeature(featureName: String, lang: Lang): Option[String] = {
    tr("common.blocks:features__decl_type_hotel", featureName, lang)
  }

  def translate(feature: FeatureValue, lang: Lang): Seq[String] = {
    def tr(keySet: String, value: String) = this.tr(keySet, value, lang).toSeq

    feature match {
      case BooleanFeatureValue(_, Some(true)) =>
        tr("common.blocks:features__decl_type_hotel", feature.feature.name)
      case IntFeatureValue(_, Some(i)) =>
        tr("common.blocks:features__decl_type_hotel", feature.feature.name)
          .map(pluralize(i))
          .filterNot(_.contains("<i18n:"))
      case EnumFeatureValue(_, Some(value)) =>
        val keySet = "common.blocks:i-" + feature.feature.name.replace("_", "-") + "-label"
        tr(keySet, value)
      case MultipleFeatureValue(_, values) =>
        val keySet = "common.blocks:i-" + feature.feature.name.replace("_", "-") + "-label"
        values.flatMap { value =>
          tr(keySet, value)
        }.toSeq
    }
  }

  private def pluralize(value: Int)(translation: String): String = {
    Translations.dynamicTranslation
      .replaceAllIn(translation, m ⇒ Speller.custom(m.group(1), m.group(2), m.group(3), addCount = false)(value))
      .replaceAllLiterally("<i18n:param>value</i18n:param>", value.toString)
  }

  def translate(hotelType: HotelType, lang: Lang): Option[String] = {
    translate("common.blocks:i-hotel-type-label", hotelType.name().toLowerCase, lang)
  }

  def isEmpty: Boolean = map.isEmpty
  def nonEmpty: Boolean = map.nonEmpty
}

object Translations extends DataDef[Translations] {
  private val dynamicTranslation = (
    """<i18n:dynamic project="tanker" keyset="dynamic" key="plural_adv">""" +
      """<i18n:count><i18n:param>value</i18n:param></i18n:count>""" +
      """<i18n:one>(.*)</i18n:one><i18n:some>(.*)</i18n:some><i18n:many>(.*)</i18n:many><i18n:none>(.*)</i18n:none>""" +
      """</i18n:dynamic>"""
    ).r

  override def dataType: DataType = DataTypes.tanker

  override def parse(is: InputStream): Translations = {
    TankerParser.parse(is)
      .ensuring(_.nonEmpty, "Translations should not be empty")
  }
}