package ru.yandex.tours.util

import java.io._
import java.nio.file.{Files, Path}
import java.util.{Timer, TimerTask}

import org.apache.commons.io.output.ByteArrayOutputStream
import org.apache.commons.io.{FileUtils, IOUtils}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.lang.Futures

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}
import scala.io.Source
import scala.util.Failure
import scala.util.control.NonFatal


/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 19.03.15
 */
object IO extends Logging {

  private val tmpDirName = sys.props("tours.tmp.dir")
  private val tmpDir =
    if ((tmpDirName eq null) || tmpDirName.isEmpty) {
      log.warn(s"Component temp directory is undefined. Provide tours.tmp.dir system property next time")
      new File(sys.props("java.io.tmpdir"), "tours_" + Randoms.nextString(6))
    } else new File(sys.props("java.io.tmpdir"), tmpDirName)

  log.info(s"Preparing temporary directory: $tmpDir")
  FileUtils.forceMkdir(tmpDir)

  {
    val list = tmpDir.listFiles()
    new Thread("tmp-cleaner") {
      override def run(): Unit = {
        log.info("Start cleaning temporary directory")
        for (file ← list) {
          try FileUtils.forceDelete(file)
          catch {
            case _: IOException ⇒ //ignore
          }
        }
        log.info("End cleaning temporary directory")
      }
    }.start()
  }

  def using[T <: Closeable, R](closeable: => T)(action: T => R): R = {
    val obj = closeable
    try action(obj)
    finally IOUtils.closeQuietly(obj)
  }

  def usingAsync[T <: Closeable, R](closeable: => T)(action: T => Future[R]): Future[R] = {
    val obj = closeable
    action(obj).andThen {
      case _ => obj.close()
    }(Futures.sameThreadExecutorContext)
  }

  def closeOnFail[T <: Closeable, R](closeable: => T)(action: T => R): R = {
    val obj = closeable
    try {
      action(obj)
    } catch {
      case NonFatal(e) => IOUtils.closeQuietly(obj); throw e
    }
  }

  def usingTmp(prefix: String)(callback: OutputStream => Unit): File = {
    val file = newTempFile(prefix)
    try {
      using(new FileOutputStream(file))(callback)
    } catch {
      case NonFatal(e) =>
        deleteFile(file)
        throw e
    }
    file
  }

  def newTempFile(prefix: String, suffix: String = "_tmp"): File = {
    File.createTempFile(prefix + "_", suffix, tmpDir)
  }

  def newTempDir(prefix: String = Randoms.nextString(5)): File = {
    val path = Files.createTempDirectory(tmpDir.toPath, prefix).toString
    new File(path)
  }

  def deleteFile(file: File): Unit = {
    if (!FileUtils.deleteQuietly(file)) log.warn(s"Can not delete file: ${file.getAbsolutePath}!")
  }

  def usingAsyncTmp(prefix: String)(callback: OutputStream => Future[Unit])
                   (implicit ec: ExecutionContext): Future[File] = {
    val file = newTempFile(prefix)
    val os = new FileOutputStream(file)
    callback(os).map(_ => file).andThen {
      case _ => os.close()
    }.andThen {
      case Failure(e) => deleteFile(file)
      case _ =>
    }
  }

  def using[T1 <: Closeable, T2 <: Closeable, R](c1: => T1, c2: => T2)(action: (T1, T2) => R): R = {
    using(c1) { obj1 =>
      using(c2) { obj2 =>
        action(obj1, obj2)
      }
    }
  }

  private val asyncCloser = new Timer("async-closer", true)

  def closeAfter[T <: Closeable](closeable: T, interval: FiniteDuration): Unit = {
    asyncCloser.schedule(new TimerTask {
      override def run(): Unit = IOUtils.closeQuietly(closeable)
    }, interval.toMillis)
  }

  def readLines(is: InputStream): Iterator[String] = Source.fromInputStream(is, "UTF-8").getLines().bindTo(is)

  def readLines(file: File): Iterator[String] = readLines(new FileInputStream(file))

  def readLines(filename: String): Iterator[String] = readLines(new File(filename))

  def writeString(callback: OutputStream => Unit): String = {
    val out = new ByteArrayOutputStream()
    using(out)(callback)
    new String(out.toByteArray)
  }

  def writeStream(callback: OutputStream => Unit): InputStream = {
    val out = new ByteArrayOutputStream()
    using(out)(callback)
    new ByteArrayInputStream(out.toByteArray)
  }

  def printStream(callback: PrintWriter => Unit): InputStream = {
    val bytes = printBytes(callback)
    new ByteArrayInputStream(bytes)
  }

  def printString(callback: PrintWriter => Unit): String = {
    val out = new StringWriter()
    using(new PrintWriter(out))(callback)
    out.toString
  }

  def printBytes(callback: PrintWriter => Unit): Array[Byte] = {
    val out = new ByteArrayOutputStream()
    using(new PrintWriter(out))(callback)
    out.toByteArray
  }

  def printFile(file: File)(callback: PrintWriter => Unit): File = {
    using(new PrintWriter(file))(callback)
    file
  }

  def printFile(filename: String)(callback: PrintWriter => Unit): File = {
    printFile(new File(filename))(callback)
  }
}
