package ru.yandex.tours.util

import java.io.{ByteArrayOutputStream, OutputStream}
import java.util.Base64
import java.util.zip.{DeflaterOutputStream, InflaterOutputStream}
import javax.crypto.spec.SecretKeySpec

import com.google.common.base.Charsets
import ru.yandex.tours.model.search.SearchProducts.Offer
import ru.yandex.tours.model.search.SearchResults.ActualizedOffer
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.util.Encryption._

import scala.util.Try

class LabelBuilder(secretKey: String) {
  private val SECRET_KEY = new SecretKeySpec(secretKey.getBytes, "AES")
  private val charset = Charsets.UTF_8

  def build(ao: ActualizedOffer, utmMark: UtmMark, userIdentifiers: UserIdentifiers): String =
    build(ao.getOffer, utmMark, userIdentifiers)

  def build(offer: Offer, utm: UtmMark, userIdentifiers: UserIdentifiers): String =
    build(offer.getHotelId.toString, offer.getSource.getOperatorId.toString, offer.getPrice, utm, userIdentifiers)

  def build(hotelId: String, operatorId: String, price: Int, utm: UtmMark, userIdentifiers: UserIdentifiers): String = {
    val values = Seq(utm.source,
      utm.medium,
      utm.campaign,
      utm.content,
      utm.term,
      userIdentifiers.uid.getOrElse(""),
      userIdentifiers.yuid.getOrElse(""),
      hotelId,
      operatorId,
      decryptSafe(utm.query),
      decryptSafe(utm.reqId),
      price.toString
    )
    encrypt(values.map(clean).mkString("\t"))
  }

  def encrypt(label: String): String = {
    val bytes = label.getBytes(charset)
    Base64.getUrlEncoder.withoutPadding().encodeToString(encryptAes(compress(bytes), SECRET_KEY))
  }
  
  def decrypt(encryptedLabel: String): String = {
    val decoder = if (encryptedLabel.contains("+") || encryptedLabel.contains("/") || encryptedLabel.contains("=")) {
      Base64.getDecoder
    } else Base64.getUrlDecoder
    new String(decompress(decryptAes(decoder.decode(encryptedLabel), SECRET_KEY)), charset)
  }

  private def decryptSafe(encryptedLabel: String): String = {
    if (encryptedLabel.isEmpty) return ""
    Try(decrypt(encryptedLabel)).getOrElse("Failed to decrypt")
  }

  private def compress(bytes: Array[Byte]): Array[Byte] = {
    pump(bytes, os => new DeflaterOutputStream(os))
  }

  private def decompress(bytes: Array[Byte]): Array[Byte] = {
    pump(bytes, os => new InflaterOutputStream(os))
  }

  private def pump(bytes: Array[Byte], pipe: OutputStream => OutputStream) = {
    val baos = new ByteArrayOutputStream()
    val os = pipe(baos)
    os.write(bytes)
    os.close()
    baos.toByteArray
  }

  private def clean(x: String) = x.replaceAll("\t", " ")
}
