package ru.yandex.tours.util

import _root_.akka.actor.ActorContext
import _root_.akka.event.{Logging => AkkaLogging}
import ch.qos.logback.classic.util.ContextInitializer
import ch.qos.logback.classic.{Logger => LogbackLogger, LoggerContext, Level}
import ch.qos.logback.core.joran.spi.JoranException
import org.slf4j.{Logger, LoggerFactory}
import sun.misc.{Signal, SignalHandler}

import scala.util.control.NonFatal

/** Mixable trait that provides logger instance.
  * For avoid boilerplate logger instance creations.
  */
trait Logging {
  protected val log = LoggerFactory.getLogger(super.getClass)
}

object Logging extends Logging with SignalHandler {

  def setLogLevel(prefix: String, level: String): Boolean = {
    val logger =
      if (prefix.isEmpty) LoggerFactory.getLogger(Logger.ROOT_LOGGER_NAME)
      else LoggerFactory.getLogger(prefix)

    logger match {
      case logger: LogbackLogger =>
        try {
          val newLevel = Level.toLevel(level)
          log.info(s"Modified logger: [${logger.getName}]. Old level: [${logger.getLevel}]. New level: [$newLevel]")
          logger.setLevel(newLevel)
          true
        } catch {
          case NonFatal(t) =>
            log.warn(s"Failed to set logging level [$level] for prefix: [$prefix]", t)
            false
        }
      case _ =>
        log.warn(s"Unknown logger class: ${logger.getClass}")
        false
    }
  }

  def setAkkaLogLevel(level: String)(implicit context: ActorContext): Boolean = {
    AkkaLogging.levelFor(level) match {
      case Some(l) =>
        context.system.eventStream.setLogLevel(l)
        true
      case _ =>
        false
    }
  }

  def openLog(): Unit = {
    try {
      val lc = LoggerFactory.getILoggerFactory.asInstanceOf[LoggerContext]

      val ci = new ContextInitializer(lc)

      lc.reset()
      ci.autoConfig()
    } catch {
      case e: JoranException =>
        e.printStackTrace();
    }
  }

  override def handle(signal: Signal): Unit = {
    val lc = LoggerFactory.getILoggerFactory.asInstanceOf[LoggerContext]
    lc.getLogger(classOf[Logging]).info("Logfile turned over due to signal " + signal.getName)

    openLog()
  }
}