package ru.yandex.tours.util

import com.codahale.metrics._

import scala.reflect.ClassTag

class UpdatableLongGauge extends Gauge[Long] {
  var value: Long = 0

  def setValue(v: Long): Unit = {
    value = v
  }

  override def getValue: Long = value
}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.03.15
 */
class Metrics(registry: MetricRegistry, group: String) {
  private def getName(name: String, names: String*) = MetricRegistry.name(group, name +: names: _*)

  def getTimer(name: String, names: String*): Timer = {
    val fullName = getName(name, names: _*)
//    getOrAdd(fullName, new Timer(new HdrHistogramResetOnSnapshotReservoir))
    //revert to this in case of any problems:
    registry.timer(fullName)
  }

  def getHistogram(name: String, names: String*): Histogram = {
    val fullName = getName(name, names: _*)
//    getOrAdd(fullName, new Histogram(new HdrHistogramResetOnSnapshotReservoir))
    //revert to this in case of any problems:
    registry.histogram(fullName)
  }

  def getMeter(name: String, names: String*): Meter = {
    registry.meter(getName(name, names: _*))
  }

  def getCounter(name: String, names: String*): Counter = {
    registry.counter(getName(name, names: _*))
  }

  def newGauge[T](name: String, names: String*)(v: => T): Gauge[T] = {
    registry.register(getName(name, names: _*), new Gauge[T] {
      override def getValue: T = v
    })
  }

  def addGauge(name: String, names: String*)(gauge: UpdatableLongGauge): Unit = {
    registry.register(getName(name, names: _*), gauge)
  }

  def registerAll(prefix: String, metrics: MetricSet): Unit = {
    import scala.collection.JavaConverters._
    for ((name, metric) <- metrics.getMetrics.asScala) {
      registry.register(getName(prefix, name), metric)
    }
  }

  private def getOrAdd[T <: Metric](fullName: String, builder: => T)(implicit classTag: ClassTag[T]): T = {
    registry.getMetrics.get(fullName) match {
      case null =>
        try {
          val metric = builder
          registry.register(fullName, metric)
        } catch {
          case e: IllegalArgumentException =>
            val metric = registry.getMetrics.get(fullName)
            if (classTag.runtimeClass.isInstance(metric)) metric.asInstanceOf[T]
            else sys.error(s"[$fullName] already registered for different type of metric: ${metric.getClass.getName}")
        }
      case metric: T => metric
    }
  }

  def apply(name: String): Metrics = new Metrics(registry, getName(name))
}

object Metrics extends Metrics(ru.yandex.common.monitoring.Metrics.defaultRegistry(), "")