package ru.yandex.tours.util

import org.apache.commons.codec.digest.DigestUtils
import org.joda.time.DateTime
import play.api.libs.json.Json
import ru.yandex.tours.avia.FlightUtils
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.search.SearchProducts.{FlightTrip, HotelSnippet, Offer}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.util.lang.Dates._

import scala.collection.JavaConverters._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 08.02.16
 */
object ReqAnsFormat {

  def sessionStart(time: DateTime): Long = {
    time.withTime(time.getHourOfDay, (time.getMinuteOfHour / 30) * 30, 0, 0).getMillis
  }

  def serpId(time: DateTime, user: UserIdentifiers, context: SearchType, request: HotelSearchRequest): String = {
    val encoded = Seq(
      sessionStart(time),
      user.yuid.getOrElse(""),
      user.uid.getOrElse(""),
      user.login.getOrElse(""),
      context,
      request.encoded
    ).mkString(":")
    DigestUtils.md5Hex(encoded)
  }

  def snippetId(snippet: HotelSnippet): String = {
    assert(snippet.getPansionsCount > 0, "snippet without pansions")
    val minPansion = snippet.getPansionsList.asScala.minBy(_.getPrice)
    val encoded = Seq(
      snippet.getHotelId,
      minPansion.getPrice,
      minPansion.getPansion.name()
    ).mkString(":")
    DigestUtils.md5Hex(encoded)
  }

  def snippet(snippet: HotelSnippet): String = {
    val json = Json.obj(
      "snippet_id" -> snippetId(snippet),
      "hotel_id" -> snippet.getHotelId,
      "price_from" -> snippet.getPriceMin,
      "start_date_from" -> snippet.getDateMin.toLocalDate.toString(),
      "start_date_to" -> snippet.getDateMax.toLocalDate.toString(),
      "nights_from" -> snippet.getNightsMin,
      "nights_to" -> snippet.getNightsMax,
      "providers" -> snippet.getSourceList.asScala.map(_.getOperatorId).distinct,
      "pansions" -> snippet.getPansionsList.asScala.map(_.getPansion.name()).distinct
    )
    json.toString()
  }

  def offer(offer: Offer): String = {
    val json = Json.obj(
      "id" -> offer.getId,
      "date" -> offer.getDate.toLocalDate.toString(),
      "nights" -> offer.getNights,
      "pansion" -> offer.getPansion.name(),
      "price" -> offer.getPrice,
      "provider" -> offer.getSource.getOperatorId,
      "free_cancellation" -> (offer.hasFreeCancellation && offer.getFreeCancellation),
      "has_free_cancellation" -> offer.hasFreeCancellation,
      "with_transfer" -> offer.getWithTransfer,
      "with_flight" -> offer.getWithFlight,
      "with_medical_insurance" -> offer.getWithMedicalInsurance
    )
    json.toString()
  }

  def trip(trip: FlightTrip): String = {
    def flightStr(flights: Seq[FlightTrip.Flight]): String =
      flights.map(f ⇒ f.getDeparture + "-" + f.getArrival).mkString(",")

    val json = Json.obj(
      "forward" → flightStr(trip.getForwardList.asScala),
      "forward_time" → FlightUtils.getFlightDuration(trip.getForwardList.asScala).getStandardMinutes,
      "backward" → flightStr(trip.getBackwardList.asScala),
      "backward_time" → FlightUtils.getFlightDuration(trip.getBackwardList.asScala).getStandardMinutes
    )
    json.toString()
  }
}
