package ru.yandex.tours.util

import java.io.{OutputStream, PrintWriter}

import ru.yandex.tours.util.collections.Bag
import ru.yandex.tours.util.lang.Futures
import ru.yandex.tours.util.parsing.Tabbed

import scala.concurrent.Future
import scala.util.{Failure, Success}

object Statistics extends Logging {
  def distribution[X, Y](name: String, objs: TraversableOnce[X], extractor: X => Y, os: OutputStream): Unit = {
    val bag = new Bag[Y]()
    objs.foreach { obj =>
      bag += extractor(obj)
    }
    printBag(name, bag, os)
  }

  def distribution[X, Y](name: String, objs: TraversableOnce[X], extractor: X => Y): Unit = {
    distribution(name, objs, extractor, System.out)
  }

  def printBag[X](name: String, bag: Bag[X], os: OutputStream): Unit = {
    val pw = new PrintWriter(os)
    pw.println(s"$name distribution:")
    bag.entriesDesc.foreach {
      case (key, value) => pw.println(Tabbed(key, value))
    }
    pw.flush()
  }

  def asyncLogTime[T](process: String, action: => Future[T]): Future[T] = {
    log.info(s"Start $process")
    val start = System.currentTimeMillis()
    action.andThen {
      case Success(_) => log.info(s"$process done in ${System.currentTimeMillis() - start} ms.")
      case Failure(e) => log.info(s"$process failed in ${System.currentTimeMillis() - start} ms. ${e.toString}")
    }(Futures.sameThreadExecutorContext)
  }
  
  def logTime[T](process: String)(action: => T): T = {
    log.info(s"Start $process")
    val start = System.currentTimeMillis()
    val result = action
    log.info(s"$process done in ${System.currentTimeMillis() - start} ms.")
    result
  }

  def printBag[X](name: String, bag: Bag[X]): Unit = printBag(name, bag, System.out)
}
