package ru.yandex.tours.util

import _root_.spray.http.StatusCodes
import _root_.akka.util.Timeout
import com.google.common.util.concurrent.RateLimiter
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.parsing.IntValue

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration._
import scala.util.Success

/**
 * hydra rate limiter client
 * https://wiki.yandex-team.ru/vertis/hydra/API#ogranichitelinagruzki/limiter
 */
class Throttler(host: String, port: Int, client: AsyncHttpClient)(implicit ec: ExecutionContext) extends Logging {
  private val metrics = Metrics("throttler")

  private val baseUrl = s"http://$host:$port/api/v2/limiter/tours/ru"
  private implicit val timeout = Timeout(1.second)

  private val localRateLimiter = RateLimiter.create(2.0d)

  def acquire(component: String, id: String): Future[Unit] = {
    tryAcquire(component, id).flatMap {
      case true => Future.successful(())
      case false => Future.failed(new ThrottledException(component, id))
    }
  }

  def tryAcquire(component: String, id: String): Future[Boolean] = {
    val url = s"$baseUrl/$component/$id"
    client.get(url).map {
      case (StatusCodes.OK, IntValue(remained)) => true
      case (StatusCodes.TooManyRequests, _) => false
      case (status, entity) =>
        log.warn(s"Unknown response from rate limiter: [$status] $entity")
        localRateLimiter.tryAcquire()
    } recover {
      case e =>
        log.error("Failed to check rate limiter", e)
        localRateLimiter.tryAcquire()
    } andThen {
      case Success(false) => metrics.getMeter(component, id).mark()
    }
  }

  def throttle[T](component: String, id: String)(action: => Future[T]): Future[T] = {
    for {
      _ <- acquire(component, id)
      res <- action
    } yield res
  }
}

class ThrottledException(component: String, id: String) extends RuntimeException(s"Request to $component/$id was throttled")