package ru.yandex.tours.util.collections

import scala.annotation.tailrec

/**
  * Created by asoboll on 07.02.17.
  */
trait AbstractMappedMap {

  protected case class Position(offset: Int, limit: Int)

  protected trait AbstractIndex[K] {
    def find(x: K): Option[Int]

    def keyIterator: Iterator[K]
  }

  protected object PositionIndex extends AbstractIndex[Int] {
    def find(x: Int): Option[Int] = Some(x)

    def keyIterator: Iterator[Int] = Iterator.from(0)
  }

  protected def emptyIndex[K]: AbstractIndex[K] = new AbstractIndex[K] {
    def find(x: K): Option[Int] = None

    def keyIterator: Iterator[K] = Iterator.empty
  }

  protected class SeqBasedIndex[K](underlying: Seq[K])
                                  (implicit ord: Ordering[K]) extends AbstractIndex[K] {
    def find(x: K): Option[Int] = {
      @tailrec
      def binSearch(start: Int, end: Int): Option[Int] = {
        if (start >= end) return None
        val pos = (end + start) / 2
        val cmp = ord.compare(underlying(pos), x)
        if (cmp == 0) return Some(pos)
        if (cmp < 0) binSearch(pos + 1, end)
        else binSearch(start, pos)
      }

      binSearch(0, underlying.size)
    }

    def keyIterator: Iterator[K] = underlying.iterator
  }

  protected sealed trait AbstractOffsets {
    def offsetFor(idx: Int): Int

    def offsetsFor(idx: Int) = Position(offsetFor(idx), offsetFor(idx + 1))

    def offsetIterator: Iterator[Int] = Iterator.from(0).map(offsetFor)

    def positionIterator: Iterator[Position] = {
      offsetIterator
        .sliding(2).map {
        case Seq(a, b) => Position(a, b)
        case _ => sys.error("Unexpected sliding result for infinite stream!")
      }
    }
  }

  protected class FixedSizeOffsets(offsetStep: Int) extends AbstractOffsets {
    def offsetFor(idx: Int): Int = idx * offsetStep
  }

  protected class SeqBasedOffsets(underlying: Seq[Int]) extends AbstractOffsets {
    def offsetFor(idx: Int): Int = underlying(idx)

    override def offsetIterator: Iterator[Int] = underlying.iterator

    def size: Int = underlying.size
  }

  protected trait AbstractData[V] {
    def get(pos: Position): V
  }

  protected trait AbstractPair[K, V] {
    protected def index: AbstractIndex[K]
    protected def offsets: AbstractOffsets
    protected def data: AbstractData[V]

    def size: Int

    def get(x: K): Option[V] = index.find(x).map(idx => data.get(offsets.offsetsFor(idx)))

    def dataIterator: Iterator[V] = offsets.positionIterator.take(size).map(data.get)

    def isFixedSizeRecords: Boolean = offsets.isInstanceOf[FixedSizeOffsets]
  }

  class MappedMap[K, V](override val size: Int,
                        protected val index: AbstractIndex[K],
                        protected val offsets: AbstractOffsets,
                        protected val data: AbstractData[V])
    extends AbstractPair[K, V] with Map[K, V] {

    override def iterator: Iterator[(K, V)] = index.keyIterator zip dataIterator

    override def +[B1 >: V](kv: (K, B1)): Map[K, B1] = throw new UnsupportedOperationException
    override def -(key: K): Map[K, V] = throw new UnsupportedOperationException
  }

  class MappedSeq[V](override val size: Int,
                     protected val offsets: AbstractOffsets,
                     protected val data: AbstractData[V])
    extends AbstractPair[Int, V] with Seq[V] {

    final protected val index = PositionIndex

    def apply(idx: Int): V = get(idx).get

    def length: Int = size

    override def iterator: Iterator[V] = dataIterator
  }

}
