package ru.yandex.tours.util.collections

import java.io.File

import org.apache.commons.io.FileUtils
import ru.yandex.tours.util.{Logging, Randoms}
import ru.yandex.vertis.curator.recipes.map.ValueSerializer

import scala.annotation.tailrec
import scala.collection.mutable
import scala.util.control.NonFatal

/**
 * Queue that dumps its content to files after reaching given threshold
 */
case class DumpingQueue[T](path: File, serializer: ValueSerializer[T], maxMemorySize: Int) extends Logging {
  require(maxMemorySize >= 0, "max memory size should be >= 0")

  FileUtils.forceMkdir(path)

  private def nextFile() = new File(
    path,
    f"dump_${System.currentTimeMillis}%020d_${Randoms.nextString(5)}"
  )

  private val lock = new Object

  private var count: Int = 0
  private var inStash: FileStash[T] = null
  private var outStash: FileStash[T] = null
  private val queue = mutable.Queue.empty[T]

  private def pushInternal(item: T): Unit = {
    count += 1
    if (outStash != null) {
      outStash.push(item)
    } else {
      queue.enqueue(item)
      if (queue.size > maxMemorySize) {
        outStash = new FileStash(nextFile(), serializer)
        outStash ++= queue
        queue.clear()
      }
    }
  }

  @tailrec
  private def pullInternal(): Option[T] = {
    if (inStash != null) {
      val out = inStash.pull()
      if (out.isDefined) out
      else {
        inStash = null
        pullInternal()
      }
    } else if (outStash != null) {
      inStash = outStash
      outStash = null
      pullInternal()
    } else if (queue.nonEmpty) {
      Some(queue.dequeue())
    } else {
      None
    }
  }

  private def loadFile(file: File): Unit = try {
    lock.synchronized {
      if (file.length() <= FileUtils.ONE_GB) {
        val stash = new FileStash[T](file, serializer)
        Iterator.continually(stash.pull())
          .takeWhile(_.isDefined)
          .flatten
          .foreach(pushInternal)
      } else {
        log.warn(s"Ignored too big file $file: ${FileUtils.byteCountToDisplaySize(file.length())}")
      }
    }
  } catch {
    case NonFatal(e) =>
  } finally {
    file.delete()
  }

  new Thread(s"dumping-queue-transfer-$path") {
    override def run(): Unit = {
      path.listFiles().iterator
        .filter(_.getName startsWith "dump_").toVector
        .sortBy(_.getName)
        .foreach(loadFile)
    }
  }.start()

  def push(item: T): Boolean = lock.synchronized {
    pushInternal(item)
    lock.notifyAll()
    true
  }

  def pull(): Option[T] = lock.synchronized {
    val out = pullInternal()
    if (out.isDefined) {
      count -= 1
    }
    out
  }

  def waitFor(): Unit = lock.synchronized {
    if (count == 0) lock.wait()
  }

  def size(): Int = lock.synchronized(count)

  def inMemorySize(): Int = lock.synchronized(queue.length)

  def isEmpty: Boolean = size() == 0
}
