package ru.yandex.tours.util.collections

import scala.collection.mutable

class Graph(edges: Iterable[(Int, Int)]) {

  private val knownVertexes = edges.flatMap { case (a, b) => Seq(a, b) }
  private val min = if (edges.nonEmpty) knownVertexes.min else 0
  private val max = if (edges.nonEmpty) knownVertexes.max else 0

  private val adjencyList = Array.fill[List[Int]](max - min + 1)(List.empty)

  private def addEdge(from: Int, to: Int) = {
    adjencyList(from) = to :: adjencyList(from)
  }

  for ((from, to) <- edges) {
    addEdge(from - min, to - min)
    addEdge(to - min, from - min)
  }

  def getConnectedComponents: Seq[Set[Int]] = {
    val visited = new mutable.BitSet(max - min)
    for {
      i <- (0 until (max - min)).iterator
      if !visited(i)
      if adjencyList(i).nonEmpty
    } yield getConnectedComponent(i, visited).map(_ + min)
  }.toSeq

  def getConnected(from: Int): Set[Int] = {
    if (edges.isEmpty) Set(from)
    else if (from < min || from > max) Set(from)
    else getConnectedComponent(from - min, mutable.BitSet.empty).map(_ + min)
  }

  /**
   *
   * @param cur - index of vertex to visit
   * @param visited - "set" of already visited vertexes. If vertex "x" is visited then visited(x) - true.
   * @return - all vertexes reachable from requested vertex
   */
  private def getConnectedComponent(cur: Int, visited: mutable.BitSet): Set[Int] = {
    visited(cur) = true
    adjencyList(cur).toSet.filterNot(visited).flatMap(getConnectedComponent(_, visited)) + cur
  }
}
