package ru.yandex.tours.util.collections

import java.io.{Closeable, File, RandomAccessFile}
import java.nio.ByteBuffer
import java.nio.channels.FileChannel
import java.util
import javax.annotation.concurrent.NotThreadSafe

import org.apache.commons.io.FileUtils
import ru.yandex.tours.util.IO

import scala.collection.mutable

/**
 * It is not true map. Removing is unsupported. All added keys should be distinct, or iterator will return duplicated keys
 */
@NotThreadSafe
class RafBasedMap[A, B](serialize: B => Array[Byte], parse: Array[Byte] => B) extends mutable.Map[A, B] with Closeable {

  private val MaxByteBufferSize = 1024 * 1024 * 1024

  private val tmpFile = IO.newTempFile("raf_map", "tmp")
  private val raf = new RandomAccessFile(tmpFile, "rw")
  private val keyMapping = mutable.Map.empty[A, Long]
  private var lastOffset = 0L
  private var closed = false
  private var buffers: Array[ByteBuffer] = null

  override def +=(kv: (A, B)): RafBasedMap.this.type = {
    if (closed) sys.error("Map is already closed")
    if (buffers ne null) sys.error("Map is already frozen")
    val (key, value) = kv
    require(!keyMapping.contains(key), s"Map already contains key $key")
    val bytes = serialize(value)
    keyMapping += key -> lastOffset
    raf.seek(lastOffset)
    raf.writeInt(bytes.length)
    raf.write(bytes)
    lastOffset += bytes.length + 4
    this
  }

  override def -=(key: A): RafBasedMap.this.type = throw new UnsupportedOperationException

  override def get(key: A): Option[B] = {
    keyMapping.get(key).map(readFromOffset)
  }

  private def readFromOffset(offset: Long): B = {
    if (closed) sys.error("Map is already closed")
    if (buffers ne null) {
      val buffer = buffers((offset / MaxByteBufferSize).toInt)
      val b = buffer.duplicate()
      b.position((offset % MaxByteBufferSize).toInt)
      val length = b.getInt
      val array = Array.ofDim[Byte](length)
      b.get(array)
      parse(array)
    } else {
      raf.seek(offset)
      val length = raf.readInt()
      val array = Array.ofDim[Byte](length)
      raf.readFully(array)
      parse(array)
    }
  }


  override def contains(key: A): Boolean = keyMapping.contains(key)

  override def keysIterator: Iterator[A] = keyMapping.keysIterator

  override def iterator: Iterator[(A, B)] = keyMapping.iterator.map {
    case (a, offset) => a -> readFromOffset(offset)
  }

  def freeze(): RafBasedMap[A, B] = {
    val channel = raf.getChannel
    val size = channel.size()
    val buffersCount = (size / MaxByteBufferSize).toInt + 1
    buffers = Array.ofDim[ByteBuffer](buffersCount)

    for (i ← 0 until buffersCount) {
      val bufferStart = i.toLong * MaxByteBufferSize.toLong
      val bufferSize = Int.MaxValue.toLong min (size - bufferStart)
      buffers(i) = channel.map(FileChannel.MapMode.READ_ONLY, bufferStart, bufferSize)
    }
    this
  }

  override def close(): Unit = {
    closed = true
    buffers = null
    raf.close()
    IO.deleteFile(tmpFile)
  }

  override def size: Int = keyMapping.size

  override def toString(): String = s"RafBasedMap($tmpFile, ${FileUtils.byteCountToDisplaySize(tmpFile.length())})"
}
