package ru.yandex.tours.util.concurrent

import java.util.concurrent.ConcurrentHashMap

import com.google.common.base.Ticker

import scala.annotation.tailrec
import scala.concurrent.Future
import scala.concurrent.duration.FiniteDuration

/**
 * Scales down bursts of work identified by user supplied key
 * '''Important! Amount of different keys should not be very large.'''
 */
class AsyncBurstScaler[K](interval: FiniteDuration, ticker: Ticker = Ticker.systemTicker()) {

  private val map = new ConcurrentHashMap[K, Long]()
  private val nanoInterval = interval.toNanos

  @tailrec
  private[concurrent] final def shouldWork(key: K, now: Long): Boolean = {
    val oldTime = map.putIfAbsent(key, now - nanoInterval)
    if (now - nanoInterval >= oldTime) {
      map.replace(key, oldTime, now) || shouldWork(key, now)
    } else {
      false
    }
  }

  def throttle(key: K)(f: => Future[Unit]): Future[Unit] = {
    val now = ticker.read()
    if (shouldWork(key, now)) f
    else Future.successful(())
  }
}
