package ru.yandex.tours.util.concurrent

import java.util.concurrent.ConcurrentLinkedQueue
import java.util.concurrent.atomic.AtomicInteger

import akka.actor.ActorSystem

import scala.concurrent.duration.{Deadline, Duration, FiniteDuration}
import scala.concurrent.{ExecutionContext, Future, Promise}

/**
 * Queue for async operations.
 * Required to limit number of concurrently running Futures
 */
class AsyncWorkQueue(maxParallelism: Int)(implicit akkaSystem: ActorSystem, ec: ExecutionContext) {
  require(maxParallelism > 0, "maxParallelism should be positive")

  /* Statuses:
    0 – waiting
    1 - started
    2 - deadline reached
   */
  private case class Task[T](f: () => Future[T], p: Promise[T], deadline: Option[Deadline]) {
    private val status = new AtomicInteger(0)
    def start(): Boolean = {
      if (status.compareAndSet(0, 1)) {
        if (deadline.exists(_.isOverdue())) {
          p.failure(new DeadlineReachedException)
          status.set(2)
          false
        } else {
          p.completeWith(f())
          true
        }
      } else {
        false
      }
    }
    def cancel(): Unit = {
      if (status.compareAndSet(0, 2)) {
        p.failure(new DeadlineReachedException)
      }
    }
  }

  private val queue = new ConcurrentLinkedQueue[Task[_]]()
  private val running = new AtomicInteger(0)

  private def tryRun1(): Boolean = {
    if (running.getAndIncrement() < maxParallelism) {
      var head = queue.poll()
      while (head ne null) {
        if (head.start()) {
          head.p.future.onComplete {
            case _ =>
              running.decrementAndGet()
              tryRun()
          }
          return true
        } else {
          head = queue.poll()
        }
      }
    }
    running.decrementAndGet()
    false
  }

  private def tryRun(): Unit = {
    while (tryRun1()) {}
  }

  /**
   * Submit future to queue.
   * Future will start when there will be free resources.
   */
  def submit[T](future: => Future[T]): Future[T] = {
    submit(future, Duration.Inf)
  }

  /**
   * Submit future to queue.
   * Future will start when there will be free resources.
   * If future not started before `deadline` returned future will be filled with DeadlineReachedException
   */
  def submit[T](future: => Future[T], deadline: Duration): Future[T] = {
    val p = Promise.apply[T]()
    val dl = deadline match {
      case fd: FiniteDuration => Some(fd.fromNow)
      case _ => None
    }
    val task = new Task(() => future, p, dl)

    deadline match {
      case fd: FiniteDuration => akkaSystem.scheduler.scheduleOnce(fd) { task.cancel() }
      case _ =>
    }

    queue.offer(task)

    tryRun()
    p.future
  }

  def clear(): Unit = {
    var h = queue.poll()
    while (h ne null) {
      h.cancel()
      h = queue.poll()
    }
  }
}

class DeadlineReachedException extends RuntimeException
