package ru.yandex.tours.util.google

import java.io.File
import java.util.concurrent.locks.ReentrantReadWriteLock

import org.apache.commons.lang.StringUtils
import ru.yandex.tours.model.BaseModel.Point
import ru.yandex.tours.util.IO

import scala.concurrent.{ExecutionContext, Future}

class CachingGooglePlaceClient(client: GooglePlaceClient, cacheFile: File)(implicit ec: ExecutionContext) extends GooglePlaceClient {
  var map: Map[(String, Option[Point]), String] = if (cacheFile.exists()) {
    readCache
  } else {
    Map.empty
  }

  private def readCache = {
    IO.readLines(cacheFile).map { line =>
      val parts = StringUtils.splitPreserveAllTokens(" @@ ")
      parts.ensuring(_.size == 4, "Invalid cache entry!")
      val lon = parts(1)
      val lat = parts(2)
      val point = if (lon.isEmpty || lat.isEmpty) {
        None
      } else {
        Some(Point.newBuilder().setLongitude(lon.toDouble).setLatitude(lat.toDouble).build())
      }
      (parts.head, point) -> parts(3)
    }.toMap
  }

  private def dumpCache() = {
    IO.printFile(cacheFile) { pw =>
      for (((request, point), response) <- map) {
        pw.println(Seq(
          request,
          point.map(_.getLongitude).getOrElse(""),
          point.map(_.getLatitude).getOrElse(""),
          response
        ).mkString(" @@ "))
      }
    }
  }

  val lock = new ReentrantReadWriteLock()

  override def search(request: String, point: Option[Point]): Future[String] = {
    val readLock = lock.readLock()
    try {
      readLock.lock()
      map.get((request, point)) match {
        case Some(ids) => Future.successful(ids)
        case None =>
          client.search(request, point).map { response =>
            val writeLock = lock.writeLock()
            try {
              writeLock.lock()
              map += (request, point) -> response
              dumpCache()
            } finally {
              writeLock.unlock()
            }
            response
          }

      }
    } finally {
      readLock.unlock()
    }
  }
}
