package ru.yandex.tours.util.hotels

import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.hotels.RichHotelSnippet
import ru.yandex.tours.model.search.SearchProducts._
import ru.yandex.tours.util.Logging

import scala.collection.JavaConverters._

object HotelSnippetUtil extends Logging {

  /** merge snippets with hotels and return RichSnippets */
  def enrich(snippets: Seq[HotelSnippet], hotelsIndex: HotelsIndex): Seq[RichHotelSnippet] = {
    val id2hotel = hotelsIndex.getHotelsById(snippets.map(_.getHotelId))
    snippets.flatMap(hs => id2hotel.get(hs.getHotelId).map(h => RichHotelSnippet(hs, h)))
  }

  def aggregate(offers: Iterable[Offer]): Iterable[HotelSnippet] = {
    offers.groupBy(offer => (offer.getWithFlight, offer.getHotelId)).map { case ((withFlight, hotelId), sameTours) =>
      val minPriceTour = sameTours.minBy(_.getPrice)
      val landing = if (minPriceTour.hasAgentBookingUrl) {
        Some(landingPage(minPriceTour))
      } else {
        None
      }
      val pansions = sameTours.groupBy(_.getPansion).map { case (pansion, samePasnionsTours) =>
        HotelSnippet.PricedPansion.newBuilder()
          .setPansion(pansion)
          .setPrice(samePasnionsTours.map(_.getPrice).min)
          .build()
      }
      val builder = HotelSnippet.newBuilder()
        .setHotelId(hotelId)
        .setWithFlight(withFlight)
        .setDateMin(sameTours.map(_.getDate).min)
        .setDateMax(sameTours.map(_.getDate).max)
        .setNightsMin(sameTours.map(_.getNights).min)
        .setNightsMax(sameTours.map(_.getNights).max)
        .setPriceMin(sameTours.map(_.getPrice).min)
        .setPriceMax(sameTours.map(_.getPrice).max)
        .setOfferCount(sameTours.size)
        .setSample(minPriceTour)
        .addAllPansions(pansions.asJava)
        .addAllSource(sameTours.map(_.getSource).toSet.asJava)
      landing.foreach(builder.setLandingPage)
      builder.build
    }
  }

  def merge(a: HotelSnippet, b: HotelSnippet): HotelSnippet = {
    require(a.getHotelId == b.getHotelId)
    require(a.getWithFlight == b.getWithFlight)

    val optLanding = resolveBy(a, b, getOptLanding)(_.getPriceMin)
    val optSample = resolveBy(a, b, getOptSample)(_.getSample.getPrice)

    val minPrice = Math.min(a.getPriceMin, b.getPriceMin)
    val maxPrice = Math.max(a.getPriceMax, b.getPriceMax)
    val count = a.getOfferCount + b.getOfferCount
    val nightsFrom = Math.min(a.getNightsMin, b.getNightsMin)
    val nightsTo = Math.max(a.getNightsMax, b.getNightsMax)
    val fromDate = Math.min(a.getDateMin, b.getDateMin)
    val toDate = Math.max(a.getDateMax, b.getDateMax)
    val pansions = mergePansions(a.getPansionsList.asScala, b.getPansionsList.asScala)
    val sources = TempSource.getSources(a, b)
    val result = HotelSnippet.newBuilder()
      .setHotelId(a.getHotelId)
      .setPriceMin(minPrice)
      .setPriceMax(maxPrice)
      .setOfferCount(count)
      .setNightsMin(nightsFrom)
      .setNightsMax(nightsTo)
      .setDateMin(fromDate)
      .setDateMax(toDate)
      .addAllPansions(pansions.asJava)
      .addAllSource(sources.asJava)
      .setWithFlight(a.getWithFlight)

    optSample.foreach(result.setSample)
    optLanding.foreach(result.setLandingPage)
    result.build()
  }

  private def landingPage(minPriceTour: Offer): HotelSnippet.LandingPage = {
    HotelSnippet.LandingPage.newBuilder()
      .setUrl(minPriceTour.getAgentBookingUrl)
      .setProviderId(minPriceTour.getSource.getOperatorId)
      .build()
  }

  private def getOptLanding(x: HotelSnippet) = if (x.hasLandingPage) Some(x.getLandingPage) else None
  private def getOptSample(x: HotelSnippet) = if (x.hasSample) Some(x.getSample) else None

  private def resolveBy[T, O, R](a: T, b: T, opt: T ⇒ Option[O])(f: T ⇒ R)(implicit ord: Ordering[R]): Option[O] = {
    import Ordering.Implicits._
    (opt(a), opt(b)) match {
      case (Some(av), Some(bv)) ⇒
        if (f(a) <= f(b)) Some(av)
        else Some(bv)
      case (None, optB) ⇒ optB
      case (optA, _) ⇒ optA
    }
  }

  private def mergePansions(a: Iterable[HotelSnippet.PricedPansion], b: Iterable[HotelSnippet.PricedPansion]) = {
    val aMap = a.map(x => x.getPansion -> x.getPrice)
    val bMap = b.map(x => x.getPansion -> x.getPrice)
    val grouped = (aMap ++ bMap).groupBy(_._1).map { case (pansion, pricedPansion) =>
      pansion -> pricedPansion.map(_._2).min
    }
    grouped.map {
      case (pansion, price) =>
        HotelSnippet.PricedPansion.newBuilder()
          .setPansion(pansion)
          .setPrice(price)
          .build
    }
  }

  private object TempSource {
    def getSources(a: HotelSnippet, b: HotelSnippet): Set[Source] =
      (getSnippetSource(a) | getSnippetSource(b)).map(_.toSource)

    private def getSnippetSource(snippet: HotelSnippet) = snippet.getSourceList.asScala.map(apply).toSet

    private def apply(source: Source): TempSource = {
      val ltRequestId = if (source.hasRequestId) Some(source.getRequestId) else None
      TempSource(source.getOperatorId,
        source.getPartnerId,
        ltRequestId)
    }
  }

  private case class TempSource(operatorId: Int, partnerId: Int, ltRequestId: Option[String]) {
    def toSource: Source = {
      val res = Source.newBuilder()
        .setOperatorId(operatorId)
        .setPartnerId(partnerId)
      ltRequestId.foreach(res.setRequestId)
      res.build()
    }
  }

}
