package ru.yandex.tours.util.http

import com.codahale.metrics.Meter
import ru.yandex.tours.util.{Metrics, Monitorings}
import ru.yandex.tours.util.http.HttpMetrics.Context
import spray.http.{StatusCode, StatusCodes}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 27.03.15
 */
class HttpMetrics(group: String, devOnly: Boolean = false,
                  warningRate: Double = 0.03, maxRate: Double = 0.9, size: Int = 1000) {
  private val metrics = Metrics(group)
  private val statusMeters: Map[Int, Meter] = (1 to 5).map(i => (i, metrics.getMeter(s"status.${i}xx"))).toMap
  private val timeoutMeter = metrics.getMeter("timeout")
  private val timer = metrics.getTimer("timing")

  private val monitorings = Monitorings(group, devOnly)
  private val errorMonitoring = monitorings.errorRate("errors", warningRate = warningRate,
    maxRate = maxRate, size = size, minErrorCount = 10)

  def start(): HttpMetrics.Context = new Context {
    val timerCtx = timer.time()
    override def finish(status: StatusCode, t: Option[Throwable]): Long = {
      getMeterForStatus(status, t).foreach(_.mark())
      timerCtx.stop()
    }
  }

  def timeout(): Unit = {
    timeoutMeter.mark()
    errorMonitoring.error()
  }

  private def getMeterForStatus(status: StatusCode, t: Option[Throwable]): Option[Meter] = {
    val statusCategory: Int = status.intValue / 100
    if (status.isSuccess) errorMonitoring.ok()
    else {
      if (t.isEmpty) errorMonitoring.error()
      else errorMonitoring.error(t.get)
    }
    statusMeters.get(statusCategory)
  }
}

object HttpMetrics {
  trait Context {
    def finish(status: StatusCode, t: Option[Throwable] = Option.empty): Long
  }
}
