package ru.yandex.tours.util.http

import java.io.{File, FileOutputStream, OutputStream}

import akka.util.Timeout
import com.ning.http.client.AsyncHandler.STATE
import com.ning.http.client.{AsyncHttpClient => NingClient, _}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.FutureTry
import ru.yandex.tours.util.lang.Futures._
import spray.http.{StatusCode, StatusCodes, Uri}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 27.03.15
 */
class NingHttpClient(proxy: Option[(String, Int)])(implicit ec: ExecutionContext) extends AsyncHttpClient with Logging {

  private val client = {
    val builder = new AsyncHttpClientConfig.Builder()
    builder
      .setAllowPoolingConnections(true)
      .setReadTimeout(2.minutes.toMillis.toInt)
      .setConnectTimeout(10.seconds.toMillis.toInt)
      .setRequestTimeout(30.seconds.toMillis.toInt)
      .setFollowRedirect(true)

    for ((host, port) <- proxy) {
      builder.setProxyServer(new ProxyServer(host, port))
    }
    new NingClient(builder.build())
  }

  private implicit class RichRequest(request: NingClient#BoundRequestBuilder) {
    def addHeaders(headers: List[(String, String)]): NingClient#BoundRequestBuilder = {
      for ((name, value) <- headers) {
        request.addHeader(name, value)
      }
      request
    }
    def executeAsync(implicit requestTimeout: Timeout): Future[Response] = {
      request.prepare.execute().asScalaFuture
    }
    def prepare(implicit requestTimeout: Timeout): NingClient#BoundRequestBuilder = {
      request.setRequestTimeout(requestTimeout.duration.toMillis.toInt)
      request
    }
  }

  override def get(url: Uri, additionalHeaders: List[(String, String)])
                  (implicit requestTimeout: Timeout): Future[(StatusCode, String)] = {
    client.prepareGet(url.toString())
      .addHeaders(additionalHeaders)
      .executeAsync
      .map { asString }
  }

  override def post(url: Uri, body: Array[Byte], additionalHeaders: List[(String, String)])
                   (implicit requestTimeout: Timeout): Future[(StatusCode, String)] = {
    client.preparePost(url.toString())
      .setBody(body)
      .addHeaders(additionalHeaders)
      .executeAsync
      .map { asString }
  }

  override def postFile(url: Uri, body: File, additionalHeaders: List[(String, String)])
                   (implicit requestTimeout: Timeout): Future[(StatusCode, String)] = {
    client.preparePost(url.toString())
      .setBody(body)
      .addHeaders(additionalHeaders)
      .executeAsync
      .map { asString }
  }

  override def put(url: Uri, body: Array[Byte], additionalHeaders: List[(String, String)])
                  (implicit requestTimeout: Timeout): Future[(StatusCode, String)] = {
    client.preparePut(url.toString())
      .setBody(body)
      .addHeaders(additionalHeaders)
      .executeAsync
      .map { asString }
  }

  override def putFile(url: Uri, file: File, additionalHeaders: List[(String, String)])
                  (implicit requestTimeout: Timeout): Future[(StatusCode, String)] = {
    client.preparePut(url.toString())
      .setBody(file)
      .addHeaders(additionalHeaders)
      .executeAsync
      .map { asString }
  }
  override def delete(url: Uri, additionalHeaders: List[(String, String)])
                     (implicit requestTimeout: Timeout): Future[(StatusCode, String)] = {
    client.prepareDelete(url.toString())
      .addHeaders(additionalHeaders)
      .executeAsync
      .map { asString }
  }

  override def getBytes(url: Uri, additionalHeaders: List[(String, String)])
                       (implicit requestTimeout: Timeout): Future[(StatusCode, Array[Byte])] = {
    client.prepareGet(url.toString())
      .addHeaders(additionalHeaders)
      .executeAsync
      .map { asBytes }
  }

  private def asBytes(response: Response) = {
    val status = StatusCodes.getForKey(response.getStatusCode)
      .getOrElse(StatusCodes.registerCustom(response.getStatusCode, response.getStatusText))
    status -> response.getResponseBodyAsBytes
  }

  private def asString(response: Response) = {
    val status = StatusCodes.getForKey(response.getStatusCode)
      .getOrElse(StatusCodes.registerCustom(response.getStatusCode, response.getStatusText))
    status -> response.getResponseBody
  }

  override def downloadToFile(url: Uri, output: File, additionalHeaders: List[(String, String)])
                             (implicit requestTimeout: Timeout): Future[StatusCode] = {

    for {
      os <- Try(new FileOutputStream(output)).toFuture
      request = client.prepareGet(url.toString()).addHeaders(additionalHeaders).prepare
      sc <- downloadFile(request.build(), os)
    } yield {
      os.close()
      sc
    }
  }

  private def downloadFile(request: Request, os: OutputStream) = {
    var sc: Option[StatusCode] = None
    client.executeRequest(request, new AsyncHandler[StatusCode] {
      override def onCompleted(): StatusCode = sc.get

      override def onThrowable(t: Throwable): Unit = {
        log.error(s"Can not download file by request $request")
        os.close()
      }

      override def onBodyPartReceived(bodyPart: HttpResponseBodyPart): STATE = {
        bodyPart.writeTo(os)
        STATE.CONTINUE
      }

      override def onStatusReceived(responseStatus: HttpResponseStatus): STATE = {
        sc = Some(responseStatus.getStatusCode)
        STATE.CONTINUE
      }

      override def onHeadersReceived(headers: HttpResponseHeaders): STATE = STATE.CONTINUE
    }).asScalaFuture
  }
}
