package ru.yandex.tours.util.http

import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.search.SearchProducts.Offer
import ru.yandex.tours.model.search.SearchResults.ActualizedOffer
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.util.LabelBuilder
import spray.http.Uri

import scala.util.Try

/**
  * Created by asoboll on 12.02.16.
  */
case class QueriesToAdd(highPriority: Seq[(String, String)], lowPriority: Seq[(String, String)]) {
  def withQuery(query: (String, String), hasHighPriority: Boolean = false): QueriesToAdd = {
    if (hasHighPriority) QueriesToAdd(highPriority :+ query, lowPriority)
    else QueriesToAdd(highPriority, lowPriority :+ query)
  }

  def getOrElse(queryName: String, defaultValue: String) =
    (lowPriority ++ highPriority).toMap.getOrElse(queryName, defaultValue)

  def toQuery = Uri.Query((lowPriority ++ highPriority).toMap)
}

case class UrlQueryPatcher(redirectUrl: String, labelBuilder: LabelBuilder,
                           queries: QueriesToAdd, label: (String, String)) {
  private def applyQueries(uri: Uri, queries: QueriesToAdd): Uri =
    uri.withQuery((queries.lowPriority ++ uri.query ++ queries.highPriority).toMap)

  def apply(url: String): String = {
    if (url.isEmpty) url
    else if (redirectUrl.isEmpty) patch(url)
    else patchRedirect(url)
  }

  private def patch(url: String): String = {
    val patchedQueries = queries.withQuery(label, hasHighPriority = true)
    Try(applyQueries(Uri(url, Uri.ParsingMode.Relaxed), patchedQueries).toString()).recover {
      case e: spray.http.IllegalUriException =>
        url + "&" + patchedQueries.toQuery.toString
    }.get
  }

  private def patchRedirect(url: String): String = {
    Try(redirect(applyQueries(Uri(url, Uri.ParsingMode.Relaxed), queries).toString())).recover {
      case e: spray.http.IllegalUriException =>
          redirect(url + "&" + queries.toQuery.toString)
    }.get
  }

  def redirect(baseUrl: String): String = {
    val encryptedUrl = labelBuilder.encrypt(baseUrl)
    val redirectQueries = QueriesToAdd(Seq.empty, Seq.empty)
      .withQuery("Label" -> label._2, hasHighPriority = true)
      .withQuery("PUrl" -> encryptedUrl)
    applyQueries(Uri(redirectUrl, Uri.ParsingMode.Relaxed), redirectQueries).toString()

  }
}

object UrlLabelerBuilder {
  val defaultSourceValue = "yandextravel"
  val defaultLabelName = "label"
  val defaultSourceName = "utm_source"
  val labelNameMap = Map(Partners.ostrovok.id -> "utm_term", Partners.ostrovokv3.id -> "utm_term").
    withDefaultValue(defaultLabelName)
  private val shouldAddSource = true

  def apply(utm: UtmMark, user: UserIdentifiers, redirectUrl: String, labelBuilder: LabelBuilder): UrlLabelerBuilder =
    if (shouldAddSource) new UrlLabelerBuilder(utm, user, redirectUrl, labelBuilder).withSource()
    else new UrlLabelerBuilder(utm, user, redirectUrl, labelBuilder)

  def labelName(offer: Offer): String = labelNameMap(offer.getSource.getPartnerId)
}

class UrlLabelerBuilder(utm: UtmMark, user: UserIdentifiers, redirectUrl: String, labelBuilder: LabelBuilder)
  extends UrlQueryPatcherBuilder {

  override def getRedirectUrl: String = redirectUrl
  override def getLabelBuilder: LabelBuilder = labelBuilder

  def buildFrom(offer: Offer, labelName: String): UrlQueryPatcher =
    UrlQueryPatcher(getRedirectUrl, getLabelBuilder, queries, labelName -> getLabelBuilder.build(offer, utm, user))

  def buildFrom(offer: Offer): UrlQueryPatcher = buildFrom(offer, UrlLabelerBuilder.labelName(offer))
  def buildFrom(ao: ActualizedOffer): UrlQueryPatcher = buildFrom(ao.getOffer)

  def buildFrom(hotelId: Int, operatorId: Int, partnerId: Int, price: Int): UrlQueryPatcher =
    this.build(UrlLabelerBuilder.labelNameMap(partnerId)
      -> getLabelBuilder.build(hotelId.toString, operatorId.toString, price, utm, user))
}

abstract class UrlQueryPatcherBuilder {
  protected var queries = QueriesToAdd(Seq.empty, Seq.empty)

  def getRedirectUrl: String

  def getLabelBuilder: LabelBuilder

  def build(label: (String, String)): UrlQueryPatcher = UrlQueryPatcher(getRedirectUrl, getLabelBuilder, queries, label)

  def withQuery(query: (String, String), highPriority: Boolean = false): this.type = {
    queries = queries.withQuery(query, highPriority)
    this
  }

  def withSource(source: String = UrlLabelerBuilder.defaultSourceValue): this.type =
    withQuery(UrlLabelerBuilder.defaultSourceName -> UrlLabelerBuilder.defaultSourceValue)
}
