package ru.yandex.tours.util.io

import java.io.{DataOutputStream, OutputStream}
import java.nio.ByteBuffer

import com.google.common.base.Charsets
import org.apache.commons.io.output.ByteArrayOutputStream

import scala.collection.mutable

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.01.16
 */
class SegmentedByteBuffer(buffer: ByteBuffer) {

  private val map = new mutable.LinkedHashMap[String, (Int, Int)]()

  while (buffer.hasRemaining) {
    val nameLen = buffer.getInt
    val nameBytes = new Array[Byte](nameLen)
    buffer.get(nameBytes)
    val name = new String(nameBytes, Charsets.UTF_8)

    val size = buffer.getInt()
    val pos = buffer.position()
    map += name -> (pos, size)
    buffer.position(buffer.position() + size)
  }

  def size: Int = map.size
  def count: Int = map.size

  def names: Seq[String] = map.keys.toSeq

  def getBuffer(name: String): ByteBuffer = {
    val (pos, size) = map(name)
    val b = buffer.duplicate()
    b.position(pos).limit(pos + size)
    b.slice()
  }

  override def toString: String =
    s"SegmentedByteBuffer(${map.map(p => p._1 + ": " + p._2._2 + " bytes").mkString(", ")})"
}

object SegmentedByteBuffer {
  def apply(byteBuffer: ByteBuffer): SegmentedByteBuffer = new SegmentedByteBuffer(byteBuffer)

  class SegmentedOutputStream(os: OutputStream) {
    private val daos = new DataOutputStream(os)
    private val names = new mutable.HashSet[String]()

    def writeSegment(name: String, bytes: Array[Byte]): Unit = {
      require(names.add(name), s"Segment with name $name already defined")
      val nameBytes = name.getBytes(Charsets.UTF_8)
      daos.writeInt(nameBytes.length)
      daos.write(nameBytes)
      daos.writeInt(bytes.length)
      daos.write(bytes)
    }

    def writeSegment(name: String)(cb: DataOutputStream => Unit): Unit = {
      val baos = new ByteArrayOutputStream()
      cb(new DataOutputStream(baos))
      writeSegment(name, baos.toByteArray)
    }

    def close(): Unit = daos.close()
  }
}