package ru.yandex.tours.util.io

import java.io.InputStream

/**
 *
 * @param iterator - Iterator of strings, which should be converted to InputStream.
 * @param is - Optional underlying input stream, which should be closed after end of iteration.
 * @param internalBufferCapacity - size of buffer, where string from iterator will be collected. Default is 1 mb
 */
class StringIteratorInputStream(iterator: Iterator[String],
                                is: Option[InputStream],
                                internalBufferCapacity: Int = 1024 * 1024) extends InputStream {

  private def next = {
    val builder = new StringBuilder((internalBufferCapacity * 1.1).toInt)
    while (builder.size < internalBufferCapacity && iterator.hasNext) {
      builder.append(iterator.next()).append("\n")
    }
    if (builder.isEmpty) {
      None
    } else {
      Some(builder.toString().getBytes)
    }
  }

  private var current = next
  private var position = 0

  override def read(): Int = {
    val a = Array.ofDim[Byte](1)
    if (read(a) == -1) {
      -1
    } else {
      a(0)
    }
  }

  override def read(b: Array[Byte], off: Int, len: Int): Int = {
    if (b == null) {
      throw new NullPointerException
    } else if (off < 0 || len < 0 || len > b.length - off) {
      throw new IndexOutOfBoundsException
    } else if (len == 0) {
      return 0
    }
    current match {
      case None => -1
      case Some(array) =>
        if (position >= array.length) {
          current = next
          position = 0
          read(b, off, len)
        } else {
          val readCount = Math.min(len, array.length - position)
          Array.copy(array, position, b, off, readCount)
          position += readCount
          readCount
        }
    }
  }

  override def close(): Unit = is.foreach(_.close())
}