package ru.yandex.tours.util.ip

import java.net.Inet4Address

import com.google.common.net.InetAddresses

import scala.collection.immutable.Stream
import scala.util.Random

/**
  * Created by asoboll on 27.12.16.
  */
trait RandomIpGenerator {

  case class Subnet(baseIp: Int, mask: Int) {
    val size: Int = (mask ^ 0xffffffff) + 1
    val base = baseIp ^ mask
  }

  def ip4fromInt(block: Subnet, x: Int): Inet4Address = {
    InetAddresses.fromInteger(block.base ^ (x | block.mask))
  }

  sealed trait Inet4AddressGenerator {
    def fromInt(x: Int): Inet4Address

    def next: Inet4Address = this.fromInt(Random.nextInt)

    def stream: Stream[Inet4Address] = Stream continually this.next
  }

  private class OneBlockGenerator(block: Subnet) extends Inet4AddressGenerator {
    override def fromInt(x: Int): Inet4Address = ip4fromInt(block, x)
  }

  private class ManyBlockGenerator(blocks: Seq[Subnet]) extends Inet4AddressGenerator {
    private val totalSize = blocks.map(_.size).sum

    override def fromInt(x: Int): Inet4Address = {
      var k = x % totalSize
      blocks.find { block =>
        k -= block.size
        k < 0
      }.fold {
        sys.error("Unexpected error while ip generation")
      } {
        ip4fromInt(_, x)
      }
    }
  }

  def create(blocks: Subnet*): Inet4AddressGenerator = {
    blocks.size match {
      case 0 => sys.error("no valid subnet to generate ip")
      case 1 => new OneBlockGenerator(blocks.head)
      case _ => new ManyBlockGenerator(blocks.sortBy(-_.size))
    }
  }
}

object RandomIpGenerator extends RandomIpGenerator
