package ru.yandex.tours.util.lang

import java.time.Month

import javax.xml.datatype.XMLGregorianCalendar

import com.sun.org.apache.xerces.internal.jaxp.datatype.XMLGregorianCalendarImpl
import org.joda.time.{DateTime, DateTimeConstants, LocalDate}
import ru.yandex.tours.model.util.proto

import scala.language.implicitConversions

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 18.02.15
 */
trait Dates {
  implicit val dateTimeOrdering = Ordering.ordered[DateTime](_.asInstanceOf[Comparable[DateTime]])
  implicit val localDateOrdering = Ordering.ordered[LocalDate](_.asInstanceOf[Comparable[LocalDate]])

  implicit def infixOrderingOps[T](x: T)(implicit ord: Ordering[T]): Ordering[T]#Ops =
    Ordering.Implicits.infixOrderingOps(x)

  implicit class RichLocalDate(date: LocalDate) {
    def toMillis: Long = proto.fromLocalDate(date)

    //can store dates up to 65535-12-31
    def toCompactInt: Int = {
      val year = date.getYear.toShort
      val monthOfYear = date.getMonthOfYear.toByte
      val dayOfMonth = date.getDayOfMonth.toByte
      (year << 16) + (monthOfYear << 8) + dayOfMonth
    }

    def nextSaturday: LocalDate = {
      date.getDayOfWeek match {
        case DateTimeConstants.SATURDAY => date.plusDays(7)
        case DateTimeConstants.SUNDAY => date.plusDays(6)
        case DateTimeConstants.MONDAY => date.plusDays(5)
        case DateTimeConstants.TUESDAY => date.plusDays(4)
        case DateTimeConstants.WEDNESDAY => date.plusDays(3)
        case DateTimeConstants.THURSDAY => date.plusDays(2)
        case DateTimeConstants.FRIDAY => date.plusDays(1)
      }
    }

    def nextWeekSaturday: LocalDate = {
      date.plusWeeks(1).withDayOfWeek(DateTimeConstants.SATURDAY)
    }

    def atMonthStart: LocalDate = date.withDayOfMonth(1)
    def atMonthEnd: LocalDate = date.plusMonths(1).withDayOfMonth(1).minusDays(1)

    def nearMonthStart(month: Month): LocalDate = {
      if (date.getMonthOfYear == month.getValue) date.atMonthStart
      else {
        val newDate = date.withMonthOfYear(month.getValue).atMonthStart
        if (newDate.isAfter(date)) newDate
        else newDate.plusYears(1)
      }
    }

    def toXMLGregorianCalendar: XMLGregorianCalendar = {
      XMLGregorianCalendarImpl.createDateTime(date.getYear, date.getMonthOfYear, date.getDayOfMonth, 0, 0, 0)
    }
  }

  implicit class RichLongDate(millis: Long) {
    def toLocalDate: LocalDate = proto.toLocalDate(millis)
  }

  implicit class RichCompactDate(int: Int) {
    def toLocalDate: LocalDate = {
      val dayOfMonth = int & 0xff
      val monthOfYear = (int >> 8) & 0xff
      val year = (int >> 16) & 0xffff
      new LocalDate(year, monthOfYear, dayOfMonth)
    }
  }

  implicit class RichXMLGregorianCalendar(calendar: XMLGregorianCalendar) {
    def toLocalDate: LocalDate = {
      new LocalDate(calendar.getYear, calendar.getMonth, calendar.getDay)
    }
  }
}

object Dates extends Dates
