package ru.yandex.tours.util.lang

import java.util.concurrent.Executor
import java.util.concurrent.atomic.AtomicInteger

import com.codahale.metrics.{Meter, Timer}
import com.google.common.util.concurrent.{ListenableFuture, MoreExecutors, SettableFuture}
import com.ning.http.client.extra.ListenableFutureAdapter
import ru.yandex.common.monitoring.error.ErrorReservoir
import ru.yandex.tours.util.Statistics

import scala.compat.java8.{FutureConverters, OptionConverters}
import scala.concurrent.duration.Duration
import scala.concurrent.{Await, ExecutionContext, Future, Promise}
import scala.util.{Failure, Success, Try}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 18.02.15
 */
trait Futures {
  private def executor(implicit ec: ExecutionContext) = ec match {
    case e: Executor => e
    case _ =>
      new Executor {
        override def execute(command: Runnable): Unit = ec.prepare().execute(command)
      }
  }

  private[util] val sameThreadExecutorContext = ExecutionContext.fromExecutorService(
    MoreExecutors.newDirectExecutorService())

  implicit class RichFuture[T](f: Future[T]) {
    def withTimerContext(timer: Timer.Context): Future[T] = {
      f.andThen {
        case _ => timer.stop()
      }(sameThreadExecutorContext)
    }
    def withErrorRateMeter(meter: Meter): Future[T] = {
      f.andThen {
        case Failure(_) => meter.mark()
      }(sameThreadExecutorContext)
    }
    def withErrorReservoir(errors: ErrorReservoir): Future[T] = {
      f.andThen {
        case Success(_) => errors.ok()
        case Failure(t) => errors.error(t)
      }(sameThreadExecutorContext)
    }
    def asGuavaFuture: ListenableFuture[T] = {
      val settable = SettableFuture.create[T]()
      f.onComplete {
        case Success(res) => settable.set(res)
        case Failure(ex) => settable.setException(ex)
      }(sameThreadExecutorContext)
      settable
    }
    def asCompletableFuture: java.util.concurrent.CompletableFuture[T] = {
      val completable = new java.util.concurrent.CompletableFuture[T]()
      f.onComplete {
        case Success(res) => completable.complete(res)
        case Failure(ex) => completable.completeExceptionally(ex)
      }(sameThreadExecutorContext)
      completable
    }
    def toUnit: Future[Unit] = f.map(_ => ())(sameThreadExecutorContext)

    def logTiming(name: String): Future[T] = {
      Statistics.asyncLogTime(name, f)
    }

    def flatten[R](implicit ev: <:<[T, Future[R]]): Future[R] = {
      f.flatMap(t => t.asInstanceOf[Future[R]])(sameThreadExecutorContext)
    }

    def await_! : T = Await.result(f, Duration.Inf)
  }

  implicit class RichListenableFuture[T](lf: ListenableFuture[T])(implicit ec: ExecutionContext) {

    /** converts guava future to scala future */
    def asScalaFuture: Future[T] = {
      val promise = Promise.apply[T]()
      lf.addListener(new Runnable {
        override def run(): Unit = promise.complete(Try(lf.get()))
      }, executor)
      promise.future
    }

    def asScalaFutureUnit: Future[Unit] = {
      val promise = Promise.apply[Unit]()
      lf.addListener(new Runnable {
        override def run(): Unit = promise.complete(Try(lf.get))
      }, executor)
      promise.future
    }
  }

  implicit class RichAsyncHttpFuture[T](lf: com.ning.http.client.ListenableFuture[T])(implicit ec: ExecutionContext) {
    /** converts guava future to scala future */
    def asScalaFuture: Future[T] = {
      ListenableFutureAdapter.asGuavaFuture(lf).asScalaFuture
    }
  }

  implicit class RichCompletableFuture[T] (cf: java.util.concurrent.CompletableFuture[T]) {
    def asScalaFuture: Future[T] = {
      FutureConverters.toScala(cf)
    }
  }

  implicit class RichOptional[T] (opt: java.util.Optional[T]) {
    def asOption: Option[T] = {
      OptionConverters.toScala(opt)
    }
  }

  def all(futures: TraversableOnce[Future[_]])(implicit ec: ExecutionContext): Future[Unit] = {
    if (futures.isEmpty) {
      Future.successful(())
    } else {
      val p = Promise[Unit]()
      val cnt = new AtomicInteger(futures.size)
      futures.foreach { f =>
        f.onComplete {
          case Success(a) =>
            val c = cnt.decrementAndGet()
            if (c == 0) {
              p.success(())
            }
          case Failure(t) => p.tryFailure(t)
        }
      }
      p.future
    }
  }

  def first[T](futures: Iterable[Future[T]])(p: T => Boolean)(implicit ec: ExecutionContext): Future[Option[T]] = {
    if (futures.isEmpty) Future.successful(None)
    else {
      val promise = Promise.apply[Option[T]]()
      futures.head.onComplete {
        case Success(res) if p(res) => promise.success(Some(res))
        case _ => promise.completeWith(first(futures.tail)(p))
      }
      promise.future
    }
  }

  def join[T](futures: Iterable[Future[T]])(implicit ec: ExecutionContext): Future[Unit] = {
    if (futures.isEmpty) {
      Future.successful(())
    } else {
      val p = Promise[Unit]()
      val cnt = new AtomicInteger(futures.size)
      futures.foreach { f =>
        f.onComplete {
          case Success(a) =>
            val c = cnt.decrementAndGet()
            if (c == 0) {
              p.success(())
            }
          case Failure(t) => p.tryFailure(t)
        }
      }
      p.future
    }
  }

  def lazyFold[T, R](futures: Iterator[Future[T]], zero: R)
                    (fold: (R, T) => R)
                    (implicit ec: ExecutionContext): Future[R] = {

    def loop(it: Iterator[Future[T]], acc: R): Future[R] = {
      val head = it.next()
      head.flatMap {
        case result if futures.hasNext => loop(it, fold(acc, result))
        case result => Future.successful(fold(acc, result))
      }
    }
    if (futures.hasNext) loop(futures, zero)
    else Future.successful(zero)
  }

  def lazySequence[T](futures: Iterator[Future[T]])(implicit ec: ExecutionContext): Future[List[T]] = {
    lazyFold(futures, List.newBuilder[T])(_ += _)
      .map(_.result())(sameThreadExecutorContext)
  }

  def lazyTraverse[F, T](seq: Iterable[F])(f: F => Future[T])(implicit ec: ExecutionContext): Future[List[T]] = {
    lazySequence(seq.iterator.map(f))
  }

  def successSequence[T](futures: Seq[Future[T]])(implicit ec: ExecutionContext): Future[Seq[T]] = {
    val tryFutures = futures.map(f => f.map(Success(_)).recover { case e => Failure(e) })
    Future.sequence(tryFutures).map(_.filter(_.isSuccess).map(_.get))
  }

  def partitionSequence[T](futures: Seq[Future[T]])(implicit ec: ExecutionContext): Future[(Seq[T], Seq[Throwable])] = {
    val tryFutures = futures.map(f => f.map(Success(_)).recover { case e => Failure(e) })
    Future.sequence(tryFutures).map { res =>
      val (successes, failures) = res.partition(_.isSuccess)
      successes.map(_.get) -> failures.map(_.failed.get)
    }
  }
}

object Futures extends Futures