package ru.yandex.tours.util

import java.util.concurrent.TimeUnit

import com.typesafe.config.Config

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration.{Duration, FiniteDuration}
import scala.util.{Failure, Success, Try}

package object lang {
  /** Appends side effects for [[scala.util.Try]] */
  implicit class SideEffectTry[A](`try`: Try[A]) {
    def onSuccess(effect: A => Unit): Try[A] = {
      `try`.foreach(effect)
      `try`
    }

    def onFailure(effect: Throwable => Unit): Try[A] = {
      `try`.failed.foreach(effect)
      `try`
    }
  }

  implicit class SideEffectOption[A](option: Option[A]) {
    def onDefined(effect: A => Unit): Option[A] = {
      option.foreach(effect)
      option
    }

    def onEmpty(effect: => Unit): Option[A] = {
      if (option.isEmpty) {
        effect
      }
      option
    }
  }

  implicit class FutureTry[A](`try`: Try[A]) {
    def toFuture: Future[A] = Future.fromTry(`try`)
  }

  private def toTryFuture[T](f: Future[T]) = {
    f.map(Success(_))(Futures.sameThreadExecutorContext)
      .recover({ case x => Failure(x) })(Futures.sameThreadExecutorContext)
  }
  
  def notFailedResults[T](futures: Iterable[Future[T]])
                         (implicit executionContext: ExecutionContext): Future[Iterable[T]] =
    Future.sequence(futures.map(toTryFuture)).map(_.flatMap(_.toOption))

  implicit class RichConfig(config: Config) {
    def getScalaDuration(path: String): Duration = {
      try {
        Duration(config.getString(path))
      } catch {
        case e: NumberFormatException =>
          FiniteDuration(config.getDuration(path, TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS)
            .toCoarsest.asInstanceOf[FiniteDuration]
      }
    }
    def getFiniteDuration(path: String): FiniteDuration = {
      try {
        val duration = Duration(config.getString(path))
        if (duration.isFinite()) duration.asInstanceOf[FiniteDuration]
        else sys.error(s"Duration is infinite at $path")
      } catch {
        case e: NumberFormatException =>
          FiniteDuration(config.getDuration(path, TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS)
            .toCoarsest.asInstanceOf[FiniteDuration]
      }
    }
  }

  def optional[T](condition: Boolean, x: T): Option[T] = if (condition) Some(x) else None

  implicit class RichOrdering[T](ord: Ordering[T]) {
    def or[U <: T](otherOrd: Ordering[U]): Ordering[U] = new Ordering[U] {
      override def compare(x: U, y: U): Int = {
        val compare = ord.compare(x, y)
        if (compare != 0) compare
        else otherOrd.compare(x, y)
      }
    }
  }
}
