package ru.yandex.common.monitoring.error

import ru.yandex.common.monitoring.util.CyclicBuffer
import com.codahale.metrics.health.HealthCheck.Result

import scala.concurrent.duration._

/**
  * Copied from ru.yandex.common.monitoring.error.ExpiringWarningErrorPercentileReservoir
  * minErrorCount added
  *
  * @author alexcrush
  */
class ExpiringPercentileReservoir(warningPercent: Int = 0,
                                  errorPercent: Int = 5,
                                  expire: FiniteDuration = 1.hour,
                                  windowSize: Int = 1000,
                                  minErrorCount: Int = 1) extends ErrorReservoir {

  require(0 <= warningPercent)
  require(warningPercent < errorPercent)
  require(errorPercent <= 100)
  require(windowSize >= 100)

  private val ttlMs = expire.toMillis

  private val results = new CyclicBuffer[TimedResult](windowSize)
  @volatile
  private var msg: String = null

  override def error(): Unit = {
    results.add(TimedResult(success = false))
  }

  override def error(msg: String): Unit = {
    error()
    this.msg = msg
  }

  override def ok(): Unit = {
    results.add(TimedResult(success = true))
  }

  override def toResult: Result = {
    import ru.yandex.common.monitoring.WarningHealthCheck._

    val from = System.currentTimeMillis() - ttlMs
    val copy = results.toSeq.filter(_.whenMs > from)
    val errors = copy.count(!_.success)
    val all = copy.size
    if (errors >= minErrorCount && (errors > all * errorPercent / 100))
      unhealthy(
        s"$errors/$all > $errorPercent% errors for the last $expire" +
          (if (msg == null) "" else s". Last: $msg."))
    else if (errors > all * warningPercent / 100)
      warning(
        s"$errors/$all > $warningPercent% errors for the last $expire" +
          (if (msg == null) "" else s". Last: $msg."))
    else
      healthy(s"$errors errors (out of $all) for the last $expire")
  }
}
