package ru.yandex.tours.util.monitoring

import com.codahale.metrics.health.HealthCheck
import com.codahale.metrics.health.HealthCheck.Result
import org.jetbrains.annotations.NotNull
import ru.yandex.common.monitoring.WarningHealthCheck
import ru.yandex.common.monitoring.error.ErrorReservoir

import scala.concurrent.duration.FiniteDuration

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.04.14 14:45
 */
class StateMonitoring(name: String) extends HealthCheck with ErrorReservoir {

  private case class State(result: Result, expire: Long) {
    def isExpired = expire > 0 && System.currentTimeMillis() > expire
  }

  @volatile
  private var state: State = State(Result.healthy(), 0L)

  def setOk(message: String) {
    state = State(Result.healthy(message), 0L)
  }

  def setOk(): Unit = setOk("")

  def setWarning(message: String) {
    state = State(WarningHealthCheck.warning(message), 0L)
  }

  def setWarning(message: String, ttl: FiniteDuration) {
    state = State(WarningHealthCheck.warning(message), System.currentTimeMillis() + ttl.toMillis)
  }

  def setCritical(message: String) {
    state = State(Result.unhealthy(message), 0L)
  }

  def setCritical(message: String, ttl: FiniteDuration) {
    state = State(Result.unhealthy(message), System.currentTimeMillis() + ttl.toMillis)
  }

  override def check(): Result = {
    if (state.isExpired) Result.healthy()
    else state.result
  }

  override def error(): Unit = setCritical("")

  override def error(msg: String): Unit = setCritical(msg)

  override def ok(): Unit = setOk()

  @NotNull
  override def toResult: Result = check()
}
