package ru.yandex.tours.util.parsing

import play.api.libs.json._

object JsonParsing {
  def parseString(value: JsValue, name: String): String = {
    value \ name match {
      case JsString(x) => x
      case JsNumber(x) => x.toString()
      case JsBoolean(x) => x.toString
      case _ => sys.error(s"Can not parse field $name from $value as string")
    }
  }

  def parseOptString(value: JsValue, name: String): Option[String] = value \ name match {
    case x: JsUndefined ⇒ None
    case JsNull ⇒ None
    case _ ⇒ Some(parseString(value, name))
  }

  def parseBoolean(value: JsValue, name: String): Boolean = value \ name match {
    case JsString(x) => x.toBoolean
    case JsBoolean(x) => x
    case _ => sys.error(s"Can not parse field $name from $value as boolean")
  }

  def parseDouble(value: JsValue, name: String): Double = value \ name match {
    case JsString(DoubleValue(x)) => x
    case JsNumber(x) => x.toDouble
    case _ => sys.error(s"Can not parse field $name from $value as double")
  }

  def parseInt(value: JsValue, name: String): Int = value \ name match {
    case JsString(IntValue(x)) => x
    case JsNumber(x) => x.toInt
    case _ => sys.error(s"Can not parse field $name from $value as int")
  }

  def parseOptInt(value: JsValue, name: String): Option[Int] = value \ name match {
    case x: JsUndefined ⇒ None
    case JsNull => None
    case _ => Some(parseInt(value, name))
  }

  def parseOptBoolean(value: JsValue, name: String): Option[Boolean] = value \ name match {
    case x: JsUndefined => None
    case JsNull => None
    case _ => Some(parseBoolean(value, name))
  }
}
