package ru.yandex.tours.util.spray

import org.slf4j.LoggerFactory
import ru.yandex.tours.util.http.HttpMetrics
import spray.http._
import spray.routing.Rejected
import spray.routing.directives.LoggingMagnet

/** Contains [[spray.routing.directives.LoggingMagnet]] for log all requests and responses to HTTP server.
  * Logging is aimed with Slf4j logger. It used logger with name 'http.access.logger',
  * so it should be defined in backed logging subsystem.
  */
class Access() {

  private val log = LoggerFactory.getLogger(getClass)
  private val accessLog = LoggerFactory.getLogger("http.access.logger")
  private val metrics = new HttpMetrics("http")
  private val metrics_ping = new HttpMetrics("http-ping")

  def getHeader(req: HttpRequest, name: String) = req.headers.find(_.is(name.toLowerCase)).map(_.value)
  def remoteIp(req: HttpRequest) = getHeader(req, "X-Forwarded-For") orElse getHeader(req, "Remote-Address") orElse getHeader(req, "x-real-ip") getOrElse "0.0.0.0"

  def showReqAndResp(req: HttpRequest) = {
    val ctx = if (req.uri.path.toString()=="/ping") metrics_ping.start() else metrics.start()
    x: Any => showResp(req, ctx)(x)
  }

  def showResp(req: HttpRequest, ctx: HttpMetrics.Context)(x: Any) {
    val ip = remoteIp(req)
    val headers = req.headers.mkString("[", ",", "]")

    val statusCode = responseToStatusCode(x)
    val elapsed = ctx.finish(statusCode) / 1000000
    accessLog.info(s"$ip ${req.method.value} ${req.uri} $headers $elapsed ${statusCode.intValue}")
  }

  def logTimeout(req: HttpRequest) = {
    metrics.timeout()
    val ip = remoteIp(req)
    accessLog.info(s"$ip ${req.method.value} ${req.uri} -1 500")
  }

  val magnet = LoggingMagnet(showReqAndResp(_))
}

